import * as React from 'react'
import {useEffect, useRef} from 'react'
import {Alert, Empty, List, Typography} from "antd"
import {useData} from "./contexts/data-context"
import {useUIState} from "./contexts/ui-state-context"
import {makeStyles, mergeClasses, shorthands} from "@griffel/react"
import type {ProcessedDataset} from "./types"
import {ChevronRightRegular} from "@fluentui/react-icons";
import {tokens} from "@fluentui/react-components";
import {getMaybeErrorMessageString} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Button} from "meteoio-ui/src/components/Button";
import {usePublicDatasetsDatasetIdGetDataset, usePublicJobSubmissionJobIdStraceOutputsStraceOutputs, usePublicDatasetsDatasetIdReleasedJobsGetReleasedJobs} from "meteoio-platform-client";
import {ItemDetailsModal} from "./item-details-modal";
import {downloadJobFile} from "../../jobs/job/downloadJobFile";
import {ArrowDownloadFilled, EyeFilled, PlayFilled} from "@fluentui/react-icons";
import {message} from "antd";
import {useNavigate} from "react-router-dom";
import {address_book} from "../../address_book";
import {Modal} from "meteoio-ui/src/components/Modal";
import {DatasetRerunForm} from "../../datasets/output/DatasetRerunForm";
import {useState} from "react";

const useStyles = makeStyles({
    container: {
        height: "100%",
        overflowY: "auto",
        display: "flex",
        flexDirection: "column",
    },
    header: {
        ...shorthands.padding("0", "14px"),
        ...shorthands.borderBottom("1px", "solid", "#f0f0f0"),
    },
    list: {
        flex: 1,
        overflowY: "auto",
        padding: '2px',
    },

    listDsGroup: {
        paddingLeft: '6px',
        paddingBottom: '4px',
        // borderBottomLeftRadius: '16px',
        borderBottomLeftRadius: '6px',

        "&:hover, &:focus-within": {
            // backgroundColor: tokens.colorNeutralBackground3Hover,//"#f0f8ff",
            backgroundColor: tokens.colorNeutralBackground2,

            '& .dsLinkArrow': {
                opacity: 0.5,
                transform: 'translate(8px, 2px)',
            }
        },
    },
    listDsGroupHeader: {
        display: 'flex',
        flexFlow: 'row no-wrap',
        columnGap: '1ch',
        alignItems: 'flex-start',
        padding: '4px',
        paddingLeft: '0',

        '& .dsLinkArrow': {
            opacity: 0,
            transform: 'translate(4px, 2px)',
            display: "inline-block",
            transition: 'opacity 0.2s ease, transform 0.2s ease',


            margin: '-12px',
            padding: '12px',
        },
    },
    dsLink: {
        display: 'block',
        fontWeight: '600',
        lineHeight: '1',
        textDecoration: "underline 1px dotted currentColor",
        color: tokens.colorBrandForegroundLink,
        padding: '3px 0 1px',

        '&:hover': {
            color: tokens.colorBrandForegroundLink,

            '& span.dsLinkArrow': {
                opacity: 1,
                transform: 'translate(12px, 2px)',
            }
        }
    },
    listDsGroupBody: {
        marginLeft: '8px',
        marginBottom: '2px',
        paddingLeft: '6px',
        paddingBottom: '4px',
        borderLeft: `1px solid ${tokens.colorNeutralStroke3}`,
        borderBottom: `1px solid ${tokens.colorNeutralStroke3}`,
        // borderBottomLeftRadius: '12px',
        borderBottomLeftRadius: '6px',
        "&:hover, &:focus-within": {
            borderLeft: `1px solid ${tokens.colorNeutralStroke2}`,
            borderBottom: `1px solid ${tokens.colorNeutralStroke2}`,
        }
    },

    listDsJobGroup: {
        paddingTop: '4px',
        paddingBottom: '4px',
        marginBottom: '2px',
        // borderBottomLeftRadius: '8px',
        borderBottomLeftRadius: '6px',
        "&:hover, &:focus-within": {
            backgroundColor: tokens.colorNeutralBackground4,
        },
    },
    listDsJobGroupHeader: {
        paddingLeft: '2px',
        display: 'flex',
        flexFlow: 'row no-wrap',
        columnGap: '1ch',
        alignItems: 'flex-start',
    },
    listCollapser: {
        // border: '1px solid #5555',
        lineHeight: '13px',
        height: '16px',
        textAlign: 'center',
        aspectRatio: 1,
        cursor: 'pointer',
        marginTop: '2px',
    },
    listCollapser_open: {
    },
    listCollapser_icon: {
        transformOrigin: 'center center',
        transition: 'transform 200ms ease',
        fontSize: '1.1em',
    },
    listCollapser_icon_open: {
        transform: 'rotate(90deg)'
    },
    jobLink: {
        color: tokens.colorBrandForegroundLink,
        fontWeight: '600',
        lineHeight: '1',
        textDecoration: "underline 1px dotted currentColor",
        padding: '2px 0 1px',
        display: 'block',
        '&:hover': {
            color: tokens.colorBrandForegroundLink,
        }
    },
    listDsJobGroupBody: {
        marginLeft: '10px',
        marginBottom: '6px',
        borderLeft: `1px solid ${tokens.colorNeutralStroke3}`,
        borderBottom: `1px solid ${tokens.colorNeutralStroke3}`,
        borderBottomLeftRadius: '6px',
        "&:hover, &:focus-within": {
            borderLeft: `1px solid ${tokens.colorNeutralStroke2}`,
            borderBottom: `1px solid ${tokens.colorNeutralStroke2}`,
        }
    },

    listItem: {
        display: 'block',
        padding: "0",
        cursor: "pointer",
        transition: "background-color 0.2s ease-in-out",
        ...shorthands.borderBottom("1px", "solid", "#f0f0f0"),

    },

    listItemDiv: {
        width: '100%',
        padding: "0 0px",
    },
    listItemHighlighted: {
        backgroundColor: tokens.colorBrandBackground2Hover,//"#e6f7ff",
    },
    listItemSelected: {
        backgroundColor: tokens.colorBrandBackground2,
        ...shorthands.borderLeft("3px", "solid", tokens.colorBrandForeground1),
    },


    ncmlLink: {
        color: tokens.colorBrandForegroundLink,
        fontWeight: '600',
        textDecoration: "underline 1px dotted currentColor",
        // padding: '2px 0 1px',
        display: 'block',
        fontSize: '0.9em',
        float: 'right',
        '&:hover': {
            color: tokens.colorBrandForegroundLink,
        }
    },
})


const CollapserBtn: React.FC<{
    isOpen?: boolean
}> = props => {
    const classes = useStyles()
    return <div className={mergeClasses(classes.listCollapser, props.isOpen && classes.listCollapser_open)}>
        <ChevronRightRegular className={mergeClasses(classes.listCollapser_icon, props.isOpen && classes.listCollapser_icon_open)} />
    </div>
}

const DatasetActionButtons: React.FC<{
    item: ProcessedDataset
}> = (props) => {
    const {item} = props
    const [messageApi, messageContextHolder] = message.useMessage({maxCount: 3})
    const {setPopupItemId} = useUIState()
    const navigate = useNavigate()
    const [isRerunModalOpen, setIsRerunModalOpen] = useState(false)

    // Get job outputs for download functionality
    const jobOutputs = usePublicJobSubmissionJobIdStraceOutputsStraceOutputs({
        pathParams: {jobId: item.jobId},
    }, {
        cacheTime: 10_000,
        staleTime: 5_000,
        enabled: true,
    })

    // Get released jobs data for rerun functionality
    const releasedJobs = usePublicDatasetsDatasetIdReleasedJobsGetReleasedJobs({
        pathParams: {datasetId: item.dsId}
    }, {
        cacheTime: 10_000,
        staleTime: 5_000,
    })

    // Find the run data for this specific job
    const runData = releasedJobs.data?.find(run => run.job_id === item.jobId)

    const handleDownloadData = async () => {
        if (!jobOutputs.data || jobOutputs.data.length === 0) {
            messageApi.error('No output files available for download')
            return
        }

        // Find the data file for this station
        const dataFile = jobOutputs.data.find(file => {
            const filename = file.open_file_path?.split('/').pop()?.toLowerCase() || ''
            return !filename.endsWith('.ncml') && 
                   filename.includes(item.locationStation?.toLowerCase() || '')
        })

        if (!dataFile?.open_file_path) {
            messageApi.error('No data file found for this station')
            return
        }

        const pathParts = dataFile.open_file_path.split('/')
        const filename = pathParts[pathParts.length - 1]
        
        try {
            await downloadJobFile(messageApi, item.jobId, dataFile.open_file_path, filename)
        } catch (error) {
            console.error('Download failed for file:', filename, error)
        }
    }

    const handleExamineMetadata = () => {
        setPopupItemId(item.id)
    }

    const handleDoRerun = () => {
        setIsRerunModalOpen(true)
    }

    return (
        <>
            {messageContextHolder}
            <div style={{ 
                display: 'flex', 
                gap: '8px', 
                marginTop: '4px', 
                marginBottom: '2px',
                alignItems: 'center'
            }}>
                <Button
                    size="small"
                    appearance="primary"
                    label="Download Data"
                    icon={<ArrowDownloadFilled />}
                    onClick={handleDownloadData}
                    disabled={jobOutputs.isLoading || !jobOutputs.data || jobOutputs.data.length === 0}
                />
                <Button
                    size="small"
                    label="Examine Metadata"
                    icon={<EyeFilled />}
                    onClick={handleExamineMetadata}
                />
                <Button
                    size="small"
                    label="Do a Rerun"
                    icon={<PlayFilled color="green"/>}
                    onClick={handleDoRerun}
                    disabled={!runData || !runData.ini_name}
                />
            </div>
            <Modal
                open={isRerunModalOpen}
                footer={null}
                onCancel={() => setIsRerunModalOpen(false)}
            >
                {runData && <DatasetRerunForm runData={runData} />}
            </Modal>
        </>
    )
}



const DsResultListItem: React.FC<{
    item: ProcessedDataset
}> = (props) => {
    const {item} = props
    const classes = useStyles()
    const {uiState, setListHighlightedDatasetId, setMapHighlightedDatasetId, setSelectedDatasetId, panTo, setPreset, setPopupItemId} = useUIState()

    const isHighlighted = uiState.listHighlightedDatasetId === props.item.id || uiState.mapHighlightedDatasetId === props.item.id
    const isHighlightedByMap = uiState.mapHighlightedDatasetId === props.item.id
    const isSelected = uiState.selectedDatasetId === props.item.id
    const itemClasses = `${classes.listItem} ${isSelected ? classes.listItemSelected : ""} ${isHighlighted ? classes.listItemHighlighted : ""}`

    const divRef = useRef<HTMLDivElement>(null)

    useEffect(() => {
        if (isHighlightedByMap && divRef.current) {
            divRef.current?.scrollIntoView({behavior: "smooth", block: 'center'})
            // NOTE: This is the leaf node --> let's acquire the selection from the map and convert it to list-sourced selection.
            setListHighlightedDatasetId(props.item.id)
            setSelectedDatasetId(props.item.id)
            setMapHighlightedDatasetId(null)
        }
    }, [props.item.id, isHighlightedByMap]);

    return <List.Item
        ref={divRef}
        className={itemClasses}
        onMouseEnter={() => setListHighlightedDatasetId(item.id)}
        onMouseLeave={() => setListHighlightedDatasetId(null)}
        onClick={() => {
            panTo([props.item.longitude, props.item.latitude])
            setPreset('split')
            setMapHighlightedDatasetId(props.item.id)
            setSelectedDatasetId(props.item.id)
        }}
    >
        <div className={classes.listItemDiv}>
            {/*<Link to={address_book.explore.view(props.item.dsId)} className={classes.dsLink}>*/}
            {/*    {props.item.title}*/}
            {/*    <span className="dsLinkArrow"><ArrowRightFilled/></span>*/}
            {/*</Link>*/}
            <div>
                {/*<div>{item.description}</div>*/}
                {props.item.locationStation &&
                    <div style={{marginTop: -2, lineHeight: '1', fontSize: "13px", color: "#444"}}>
                        Station: {props.item.locationStation}
                        {/*<div style={{float: 'right'}}>
                            <Button
                                size="small"
                                label="See details"
                                onClick={() => {
                                    setPopupItemId(props.item.id)
                                }}
                            />
                        </div>*/}
                    </div>}
                {props.item.variables.length > 0 && (
                    <div style={{marginTop: 1, marginBottom: 0, lineHeight: '1', fontSize: "11px", color: "#777"}}>
                        Variables: {props.item.variables.slice(0, 30).join(", ")}
                        {props.item.variables.length > 30 && "..."}
                    </div>
                )}
                {isSelected && <DatasetActionButtons item={props.item} />}
            </div>
        </div>
    </List.Item>;
};


const DsJobResultList: React.FC<{
    resultList: ProcessedDataset[]
    dsId: string
    jobId: string
}> = props => {
    const classes = useStyles()

    const title = props.resultList?.[0]?.title // TODO: What metadata should be used?

    const [isOpen, setIsOpen] = React.useState<boolean>(false)

    const {uiState} = useUIState()

    const isHighlightedByMap = props.resultList.some(item => uiState.mapHighlightedDatasetId === item.id)
    const divRef = useRef<HTMLDivElement>(null)

    useEffect(() => {
        if (isHighlightedByMap && divRef.current) {
            setIsOpen(true)
            // NOTE: let the leaf node acquire the selection and do the scrolling
            // setListHighlightedDatasetId(uiState.mapHighlightedDatasetId)
            // setMapHighlightedDatasetId(null)
            // setTimeout(() => {
            //     divRef.current?.scrollIntoView({behavior: "smooth"})
            // }, 10)
        }
    }, [uiState.mapHighlightedDatasetId, isHighlightedByMap]);

    return <div className={classes.listDsJobGroup} ref={divRef}>
        <div className={classes.listDsJobGroupHeader} onClick={() => setIsOpen(v => !v)}>
            <CollapserBtn isOpen={isOpen}/>
            {/*<Link to={address_book.jobs.view(props.jobId)} className={classes.jobLink}>*/}
            <span style={{cursor: 'pointer'}}>{title}</span>
                {/*<span className="dsLinkArrow"><ArrowRightFilled/></span>*/}
            {/*</Link>*/}
        </div>
        {isOpen && <div className={classes.listDsJobGroupBody}>
            <List
                dataSource={props.resultList}
                size="small"
                // split
                // bordered
                // pagination={{}}
                renderItem={(item) => <DsResultListItem item={item}/>}
            />
        </div>}
    </div>
}

const DsResultList: React.FC<{
    resultList: ProcessedDataset[]
    dsId: string
}> = props => {
    const classes = useStyles()

    const dsDataQuery = usePublicDatasetsDatasetIdGetDataset({pathParams: {datasetId: props.dsId}}, {cacheTime: 10_000, staleTime: 5_000})

    const title = dsDataQuery?.data?.heading?.title

    const jobIds = React.useMemo<string[]>(() => {
        const jobIds = new Set<string>()
        props.resultList.forEach(item => {
            jobIds.add(item.jobId)
        })
        return [...jobIds].sort().toReversed()
    }, [props.resultList])

    const [isOpen, setIsOpen] = React.useState<boolean>(false)

    const {uiState} = useUIState()

    const isHighlightedByMap = props.resultList.some(item => uiState.mapHighlightedDatasetId === item.id)
    const divRef = useRef<HTMLDivElement>(null)

    useEffect(() => {
        if (isHighlightedByMap && divRef.current) {
            setIsOpen(true)
            // NOTE: let the leaf node acquire the selection and do the scrolling
            // setTimeout(() => {
            //     divRef.current?.scrollIntoView({behavior: "smooth"})
            // }, 10)
        }
    }, [uiState.mapHighlightedDatasetId, isHighlightedByMap]);

    return <div className={classes.listDsGroup} ref={divRef}>
        <div className={classes.listDsGroupHeader} onClick={() => setIsOpen(v => !v)}>
            <CollapserBtn isOpen={isOpen}/>
            {/*<Link to={address_book.explore.view(props.dsId)} className={classes.dsLink}>*/}
            <span style={{cursor: 'pointer'}}>{title}</span>
                {/*<span className="dsLinkArrow"><ArrowRightFilled/></span>*/}
            {/*</Link>*/}
        </div>
        {isOpen && <div className={classes.listDsGroupBody}>
            {jobIds.map(jobId =>
                <DsJobResultList key={jobId} dsId={props.dsId} jobId={jobId} resultList={props.resultList.filter(item => item.jobId === jobId)}/>)}
        </div>}
    </div>
}

const RootResultList: React.FC<{
    resultList: ProcessedDataset[]
}> = props => {
    const datasetIds = React.useMemo<string[]>(() => {
        const datasetIds = new Set<string>()
        props.resultList.forEach(item => {
            datasetIds.add(item.dsId)
        })
        return [...datasetIds].sort()
    }, [props.resultList])
    return <>
        {datasetIds.map(dsId =>
            <DsResultList key={dsId} dsId={dsId} resultList={props.resultList.filter(item => item.dsId === dsId)}/>)}
    </>
}

export function DatasetList() {
    const classes = useStyles()
    const {filteredDatasets, isLoading, isError, error} = useData()

    return (
        <div className={classes.container}>
            <div className={classes.header}>
                <Typography.Title level={4} style={{marginTop: 0}}>
                    {filteredDatasets.length} datasets found
                </Typography.Title>
            </div>
            <div className={classes.list}>
                {filteredDatasets.length > 0 ? (
                    <RootResultList resultList={filteredDatasets}/>
                ) : (
                    isError ? (
                        <Alert
                            message="Error loading data"
                            description={getMaybeErrorMessageString(error)}
                            type="error"
                            style={{margin: "16px"}}
                            showIcon
                        />
                    ) : (
                        <Empty
                            description="No datasets match your criteria. Try adjusting the filters."
                            style={{padding: "40px"}}
                        />
                    )
                )}
            </div>
            <ItemDetailsModal/>
        </div>
    )
}
