import * as React from 'react'
import {createContext, type ReactNode, useContext, useMemo, useState} from 'react'

type Preset = "split" | "list" | "map"

interface UIState {
    listHighlightedDatasetId: string | null
    mapHighlightedDatasetId: string | null
    selectedDatasetId: string | null
    panToCoordinates: [number, number] | null
    showList: boolean
    showMap: boolean
    preset: Preset
    popupItemId?: string | null
}

interface UIContextType {
    uiState: UIState
    setListHighlightedDatasetId: (id: string | null) => void
    setMapHighlightedDatasetId: (id: string | null) => void
    setSelectedDatasetId: (id: string | null) => void
    panTo: (coords: [number, number]) => void
    clearPanRequest: () => void
    setPreset: (preset: Preset) => void
    setPopupItemId: (id: string | null) => void
}

const UIStateContext = createContext<UIContextType | undefined>(undefined)

const initialState: UIState = {
    /** Highlighting sourced by list component */
    listHighlightedDatasetId: null,
    /** Highlighting sourced by map component */
    mapHighlightedDatasetId: null,
    /** Selected dataset (persistent until another is selected) */
    selectedDatasetId: null,
    panToCoordinates: null,
    showList: true,
    showMap: true,
    preset: "split",
    popupItemId: null,
}

export const UIStateProvider = ({children}: { children: ReactNode }) => {
    const [uiState, setUiState] = useState<UIState>(initialState)

    const value = useMemo<UIContextType>(() => {

        const setListHighlightedDatasetId = (id: string | null) => {
            setUiState((prev) => ({...prev, listHighlightedDatasetId: id}))
        }
        const setMapHighlightedDatasetId = (id: string | null) => {
            setUiState((prev) => ({...prev, mapHighlightedDatasetId: id}))
        }

        const setSelectedDatasetId = (id: string | null) => {
            setUiState((prev) => ({...prev, selectedDatasetId: id}))
        }

        const panTo = (coords: [number, number]) => {
            setUiState((prev) => ({...prev, panToCoordinates: coords}))
        }

        const clearPanRequest = () => {
            setUiState((prev) => ({...prev, panToCoordinates: null}))
        }

        const setPreset = (preset: Preset) => {
            setUiState((prev) => ({
                ...prev,
                preset,
                showList: preset === "split" || preset === "list",
                showMap: preset === "split" || preset === "map",
            }))
        }

        const setPopupItemId = (id: string | null) => {
            setUiState((prev) => ({...prev, popupItemId: id}))
        }

        return {
            uiState,
            setListHighlightedDatasetId,
            setMapHighlightedDatasetId,
            setSelectedDatasetId,
            panTo,
            clearPanRequest,
            setPreset,
            setPopupItemId,
        }
    }, [uiState])

    return <UIStateContext.Provider value={value}>{children}</UIStateContext.Provider>
}

export const useUIState = () => {
    const context = useContext(UIStateContext)
    if (context === undefined) {
        throw new Error("useUIState must be used within a UIStateProvider")
    }
    return context
}
