import * as React from 'react'
import {createContext, type ReactNode, useCallback, useContext, useMemo, useState} from 'react'
import {extractUniqueValues, processDatasetEntry} from "../utils/data-processing"
import type {FilterState, ProcessedDataset} from "../types"
import dayjs from "dayjs"
import {usePublicDatasetsNcmlSearchSearchDatasets} from "meteoio-platform-client";

interface DataContextType {
    // Data
    allDatasets: ProcessedDataset[]
    filteredDatasets: ProcessedDataset[]

    // Filters
    filters: FilterState
    updateFilters: (newFilters: Partial<FilterState>) => void

    // Unique values for filter options
    uniqueValues: {
        variables: string[]
        keywords: string[]
        creators: string[]
        sources: string[]
        locations: string[]
    }

    // Loading and pagination
    isLoading: boolean
    isError: boolean
    error: any
    totalCount: number
    refetch: () => void
}

const DataContext = createContext<DataContextType | undefined>(undefined)

export const DataProvider = ({children}: { children: ReactNode }) => {
    const [filters, setFilters] = useState<FilterState>({
        searchTerm: "",
        dateRange: [null, null],
        variables: [],
        keywords: [],
        creators: [],
        sources: [],
        locations: [],
        limit: 50,
        offset: 0,
    })

    // Prepare query variables
    const queryVariables = useMemo(() => {
        const queryParams: any = {
            limit: filters.limit,
            offset: filters.offset,
        }

        if (filters.searchTerm) {
            queryParams.text_query = filters.searchTerm
        }

        if (filters.dateRange[0] && filters.dateRange[1]) {
            queryParams.time_start = dayjs(filters.dateRange[0]).toISOString()
            queryParams.time_end = dayjs(filters.dateRange[1]).toISOString()
        }

        return {queryParams}
    }, [filters])

    // Use the TanStack Query hook
    const {
        data: rawResponse,
        isLoading,
        isError,
        error,
        refetch,
    } = usePublicDatasetsNcmlSearchSearchDatasets(queryVariables, {
        staleTime: 10 * 1000,
        refetchOnWindowFocus: false,
    })

    // Process raw data into usable format
    const allDatasets = useMemo(() => {
        if (!rawResponse?.items) return []

        return rawResponse.items.map(processDatasetEntry).filter((dataset): dataset is ProcessedDataset => dataset !== null)
    }, [rawResponse])

    // Extract unique values for filter dropdowns -- Actually: empty sets. -- Will not provide dropdown autocompletion yet.
    const uniqueValues = useMemo(() => {
        return extractUniqueValues()
    }, [])

    // Apply client-side filtering for filters not handled by the API
    // ...? Actually, nope. It will let the server implement the filters.
    const filteredDatasets = allDatasets
    // const filteredDatasets = useMemo(() => {
    //     return allDatasets.filter((dataset) => {
    //         // Variables filter
    //         if (filters.variables.length > 0 && !filters.variables.some((v) => dataset.variables.includes(v))) {
    //             return false
    //         }
    //
    //         // Keywords filter
    //         if (filters.keywords.length > 0 && !filters.keywords.some((k) => dataset.keywords.includes(k))) {
    //             return false
    //         }
    //
    //         // Creators filter
    //         if (filters.creators.length > 0 && dataset.creatorName && !filters.creators.includes(dataset.creatorName)) {
    //             return false
    //         }
    //
    //         // Sources filter
    //         if (filters.sources.length > 0 && dataset.source && !filters.sources.includes(dataset.source)) {
    //             return false
    //         }
    //
    //         // Locations filter
    //         if (filters.locations.length > 0) {
    //             const hasLocation =
    //                 (dataset.location && filters.locations.includes(dataset.location)) ||
    //                 (dataset.locationStation && filters.locations.includes(dataset.locationStation))
    //             if (!hasLocation) {
    //                 return false
    //             }
    //         }
    //
    //         return true
    //     })
    // }, [allDatasets, filters])

    const updateFilters = useCallback((newFilters: Partial<FilterState>) => {
        setFilters((prev) => ({...prev, ...newFilters}))
    }, [])

    const value = {
        allDatasets,
        filteredDatasets,
        filters,
        updateFilters,
        uniqueValues,
        isLoading,
        isError,
        error,
        totalCount: rawResponse?.total || 0,
        refetch,
    }

    return <DataContext.Provider value={value}>{children}</DataContext.Provider>
}

export const useData = () => {
    const context = useContext(DataContext)
    if (context === undefined) {
        throw new Error("useData must be used within a DataProvider")
    }
    return context
}
