import React, {useCallback, useDeferredValue, useEffect, useState} from "react";
import {useDatasetConfigTxHooks} from "./useDatasetConfigTxHooks";

const _STOP_SIGNALS: Set<string> = new Set<string>()

/** Use case: prevent pushing accumulated updates after a delete command has been sent, to avoid failed update errors */
export function stopPushes(key: string) {
    _STOP_SIGNALS.add(key)
    console.debug('useDatasetConfigTxDraftPusher stopped key', key)
}
export function unstopPushes(key: string) {
    _STOP_SIGNALS.delete(key)
    console.debug('useDatasetConfigTxDraftPusher unstopped key', key)
}

export function useDatasetConfigTxDraftPusher<T>(key: string, pusher?: (draft?: T) => Promise<void>, preloader?: () => Promise<T>): [
    T,
    React.Dispatch<React.SetStateAction<T>>,
    {
        error: unknown,
        isLoading: boolean,
        reload: () => Promise<void>,
    }
] {
    const [isLoading, setIsLoading] = useState<boolean>(true)
    const [error, setError] = useState()
    const [draft, setDraft] = useState<T | undefined>()

    const draft_deferred = useDeferredValue(draft)

    // store state on session storage so that we later avoid depending on the state for hooked actions
    useEffect(() => {
        if (draft_deferred) {
            sessionStorage?.setItem?.(key, JSON.stringify(draft_deferred))
        }
    }, [key, draft_deferred])

    // load draft from server (at first key-based mount or at explicit call)
    const reload = useCallback(async () => {
        setIsLoading(true)
        try {
            const data = await preloader()
            setDraft(data)
        } catch (e) {
            setError(e)
        } finally {
            setIsLoading(false)
        }
    }, [key])

    useEffect(() => {
        // NOTE: When the key changes, it's the case that the dataset config transaction ID has changes and the draft data must be invalidated.
        //       No draft is lost unexpectedly as key changes can only occur after voluntary user actions like transaction commit/discard.
        // if (draft === undefined) {
        setDraft(undefined)
        reload().then();
        // }
    }, [key])

    // define push from sessionStorage to avoid depending on the state
    const push_ = useCallback(async () => {
        if (!pusher) {
            return
        }
        if (_STOP_SIGNALS.has(key)) {
            console.debug('useDatasetConfigTxDraftPusher not pushing stopped key', key)
            return
        }
        const draft_str = sessionStorage?.getItem?.(key)
        if (draft_str) {
            const draft = JSON.parse(draft_str)
            await pusher(draft)
            // sessionStorage?.removeItem?.(key)
        }
    }, [key])

    useDatasetConfigTxHooks({
        // push before tx application
        onBeforeApply: async () => {
            await push_()
            sessionStorage?.removeItem?.(key)
        },

        // avoid pushing after tx discard
        onBeforeDiscard: async () => {
            sessionStorage?.removeItem?.(key)
            // TODO?: setDraft(null)   // This may fix the cache invalidation issue [1ec14e60-f61a-40b4-a4b2-77209cac2169]
        }
    }, [key])

    // push before abrupt window close
    useEffect(() => {
        const h = () => {
            //console.debug('useDatasetConfigTxDraftPusher', 'window beforeunload')
            push_().then()
        }
        window?.addEventListener?.('beforeunload', h)
        return () => {
            //console.debug('useDatasetConfigTxDraftPusher', 'key deactivation')
            push_().then(() => {
                window?.removeEventListener('beforeunload', h)
            })
        }
    }, [key])

    return [
        draft,
        setDraft,
        {error, isLoading, reload},
    ]
}
