import * as React from "react";
import {useMemo, useState} from "react";
import {useDatasetConfigTx} from "../../useDatasetConfigTx";
import {DsRevisionDiff} from "./DsRevisionDiff";
import {useParams} from "react-router-dom";
import {BACKEND_CONFIG, DatasetRevisionsListEntry,} from "meteoio-platform-client";
import {ButtonProps} from "meteoio-ui/src/components/Button";
import {Modal} from "meteoio-ui/src/components/Modal";
import {DsRevRestorer} from "./DsRevRestorer";
import {DsRevClearContents} from "./DsRevClearContents";

interface Action {
    buttonProps: ButtonProps
    // No async actions.
}

export interface DiffOption extends Action {
    diffProps: React.ComponentProps<typeof DsRevisionDiff>
    compareWith_label: React.ReactNode  // Used for menu section that already has label "Compare with..."
    isForPreCommitCheck?: boolean  // Only one option should have this flag
}

interface IUseDsRevDiffOptions {
    diffOptions: DiffOption[]
    otherActions: Action[]
    renderNode?: React.ReactNode
}


const ENABLE_EXPORT: boolean = false


export const useDsRevActions = (rev?: DatasetRevisionsListEntry, isLatest?: boolean, previousRevision?: DatasetRevisionsListEntry): IUseDsRevDiffOptions => {

    // TODO: We need to know the head version ID to avoid relying on isLatest.

    const {datasetId} = useParams();
    const [workingTxId, {isHead /*, setConfigTxId*/}] = useDatasetConfigTx(datasetId);

    const [showDiff, setShowDiff] = useState<null | React.ComponentProps<typeof DsRevisionDiff>>(null)

    const [showRestoring, setShowRestoring] = useState<null | { restoreRev: DatasetRevisionsListEntry }>(null)
    const [showClearing, setShowClearing] = useState<null | { clearRev: DatasetRevisionsListEntry }>(null)

    /////////////////////////////////////////////////////////////////////////////////////////////

    const diffOptions = useMemo<DiffOption[]>(() => {
        // Build the array of options conditionally based on `rev` and state

        if (!rev || rev.emptied_at) {
            return []
        }

        const diffOptions: DiffOption[] = []

        if (isHead) {
            // There's no pending version

            if (rev.id === workingTxId || isLatest) {
                // NOTE: workingTxId might be != rev.id while still referring to it as the head version (`head` or null values)
                // No relevant comparison
            } else {
                // The rev argument can be compared with the head version (which is the current version with no ambiguity with no pending version)
                diffOptions.push({
                    buttonProps: {
                        label: `Compare with Current version`,
                    },
                    compareWith_label: `Current version`,
                    diffProps: {
                        leftTxId: rev.id,
                        rightTxId: "head",
                        leftLabel: <>Version {rev.number === null ? "(to be assigned)" : rev.number}</>,
                        rightLabel: <>Current version</>,
                    },
                });
            }
        } else {
            // else: There's a pending revision

            if (rev.id === workingTxId) {
                // It's the pending rev -- Note that in this case the head version is on the left.
                diffOptions.push({
                    buttonProps: {
                        label: `View pending changes`,
                    },
                    compareWith_label: `Confirmed version`,
                    diffProps: {
                        leftTxId: 'head',
                        rightTxId: workingTxId ?? rev.id,  // it's the same!
                        leftLabel: <>Current Confirmed version</>,
                        rightLabel: <>Pending revision</>,
                    },
                    isForPreCommitCheck: true,
                });
            } else {
                // It's not the pending rev, so the rev argument can be compared with the pending rev or the confirmed rev.
                diffOptions.push({
                    buttonProps: {
                        label: `Compare with Current Confirmed version`,
                    },
                    compareWith_label: `Confirmed version`,
                    diffProps: {
                        leftTxId: rev.id,
                        rightTxId: 'head',
                        leftLabel: <>Version {rev.number === null ? "(to be assigned)" : rev.number}</>,
                        rightLabel: <>Current Confirmed version</>,
                    },
                });
                diffOptions.push({
                    buttonProps: {
                        label: `Compare with Pending revision`,
                    },
                    compareWith_label: `Pending revision`,
                    diffProps: {
                        leftTxId: rev.id,
                        rightTxId: workingTxId,
                        leftLabel: <>Version {rev.number === null ? "(to be assigned)" : rev.number}</>,
                        rightLabel: <>Pending revision</>,
                    },
                });
            }
        }

        if (previousRevision && rev.id) {
            diffOptions.push({
                buttonProps: {
                    label: `Compare with Previous version`,
                },
                compareWith_label: `Previous version`,
                diffProps: {
                    leftTxId: previousRevision.id,
                    rightTxId: rev.id,
                    leftLabel: <>Version {previousRevision.number === null ? "(to be assigned)" : previousRevision.number}</>,
                    rightLabel: <>Version {rev.number === null ? "(to be assigned)" : rev.number}</>,
                },
            });

        }

        diffOptions.forEach(opt => {
            opt.buttonProps.onClick = () => setShowDiff(opt.diffProps)
        })

        return diffOptions
    }, [rev, isLatest, workingTxId, isHead, previousRevision])

    /////////////////////////////////////////////////////////////////////////////////////////////

    const otherActions = useMemo<Action[]>(() => {
        if (!rev) {
            return []
        }

        const otherActions: Action[] = []

        if (rev.id !== workingTxId && !rev.emptied_at) {
            otherActions.push({
                buttonProps: {
                    label: <>Restore this configuration</>,
                    onClick: () => setShowRestoring({restoreRev: rev})
                }
            })
        }

        if (rev.id !== workingTxId && !isLatest && !rev.emptied_at) {
            otherActions.push({
                buttonProps: {
                    label: <>Clear this configuration</>,
                    onClick: () => setShowClearing({clearRev: rev})
                }
            })
        }

        if (ENABLE_EXPORT) {
            otherActions.push({
                buttonProps: {
                    href: `${BACKEND_CONFIG.baseUrl}/internal/datasets/${datasetId}/config_txs/${rev.id}/export`,
                    // download: "RevisionExport",
                    label: <>Export binary file</>,
                }
            })
        }

        // if (rev.id !== workingTxId && !isLatest && !rev.emptied_at) {
        //     // TODO: WARNING: This is partially working -- the UI requires many adaptions, otherwise the loaded version looks like a pending version (!isHead)...
        //     otherActions.push({
        //         buttonProps: {
        //             label: <>Explore this version contents</>,
        //             onClick: () => setConfigTxId(rev.id)
        //         }
        //     })
        // }

        return otherActions
    }, [rev, isLatest, workingTxId, isHead])

    const renderNode = <>
        {showDiff && <Modal
            title="Dataset Configurartion Differences"
            footer={null}
            onCancel={() => setShowDiff(null)}
            open={!!showDiff}
            width={800}
        >
            <DsRevisionDiff {...showDiff} />
        </Modal>}
        {showRestoring && <DsRevRestorer restoreRev={showRestoring.restoreRev} onDismiss={() => setShowRestoring(null)}/>}
        {showClearing && <DsRevClearContents clearRev={showClearing.clearRev} onDismiss={() => setShowClearing(null)}/>}
    </>

    return {
        diffOptions,
        otherActions,
        renderNode,
    }
}
