import React, {useMemo} from 'react';
import {Diff, FileData, Hunk, parseDiff} from 'react-diff-view';
import 'react-diff-view/style/index.css';
import {useInternalDatasetsDatasetIdConfigTxsTxIdDiffFromOtherTxIdGetConfigDiff} from "meteoio-platform-client";
import {useParams} from "react-router-dom";
import {usePromise} from "meteoio-ui/src/utils/usePromise";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {Collapse} from "antd";
import {makeStyles, shorthands, tokens} from "@fluentui/react-components";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {ArrowRightRegular, BinRecycleRegular, DocumentAddRegular, DocumentEditRegular} from "@fluentui/react-icons";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Alert} from "meteoio-ui/src/components/Alert";


const useStyles = makeStyles({
    diffCollapse: {
        '& > .ant-collapse-item > .ant-collapse-content > .ant-collapse-content-box': {
            ...shorthands.padding(0),
        }
    },
})


export const DsRevisionDiff: React.FC<{
    leftTxId: string
    rightTxId: string
    leftLabel?: React.ReactNode
    rightLabel?: React.ReactNode
}> = props => {
    const styles = useStyles()

    const {datasetId} = useParams()
    const diff_query = useInternalDatasetsDatasetIdConfigTxsTxIdDiffFromOtherTxIdGetConfigDiff({
        pathParams: {
            datasetId,
            txId: props.rightTxId,
            otherTxId: props.leftTxId,
        }
    })

    const [diff_text, {error, isLoading}] = usePromise<string>(() => {
        // @ts-ignore
        return diff_query?.data?.text?.() ?? ''
    }, [diff_query?.data])

    const diff = useMemo<FileData[]>(() => {
        if (diff_text) {
            return parseDiff(diff_text)
        }
        return []
    }, [diff_text]);
    console.debug({diff})

    if (isLoading || diff_query?.isLoading) {
        return <Spinner tall/>
    }

    return <>
        <MaybeErrorAlert error={diff_query?.error ?? error}/>
        {diff?.length === 0 && <Alert level="info">No configuration differences</Alert>}
        {diff?.map?.(r => r.similarity)}
        {diff?.length > 0 && <>
            <div style={{display: 'grid', gridTemplateColumns: '1fr 3ch 1fr', textAlign: 'center', margin: '1em 0'}}>
                <div style={{background: tokens.colorNeutralBackground3}}>
                    {props.leftLabel}
                </div>
                <div>
                    <ArrowRightRegular style={{position: 'relative', top: 2}}/>
                </div>
                <div style={{background: tokens.colorNeutralBackground3}}>
                    {props.rightLabel}
                </div>
            </div>
            <Collapse
                defaultActiveKey={diff.map((fd, i) => `${i}`)}
                destroyInactivePanel
                rootClassName={styles.diffCollapse}
                items={diff.map((fd: FileData, i) => {
                    return ({
                        key: `${i}`,
                        label: <Stack horizontal alignItems="center" justifyContent="space-between">
                            <Stack horizontal alignItems="center">
                                {/* Icon and colors based on the type of diff */}
                                {fd.type === 'add' &&
                                    <DocumentAddRegular
                                        style={{color: tokens.colorStatusSuccessForeground1}}
                                        fontSize={20}
                                        title="add"
                                    />}
                                {fd.type === 'modify' &&
                                    <DocumentEditRegular
                                        style={{color: tokens.colorPaletteYellowForeground1}}
                                        fontSize={20}
                                        title="modify"
                                    />}
                                {fd.type === 'delete' &&
                                    <BinRecycleRegular
                                        style={{color: tokens.colorStatusDangerForeground1}}
                                        fontSize={20}
                                        title="delete"
                                    />}
                                <span>
                                    {fd.oldPath === fd.newPath
                                        ? fd.oldPath
                                        : (fd.type === 'add'
                                                ? fd.newPath
                                                : (fd.type === 'delete'
                                                        ? fd.oldPath
                                                        : (fd.type === 'rename'
                                                                ? `${fd.oldPath} → ${fd.newPath}`
                                                                : (fd.type === 'copy'
                                                                        ? `${fd.oldPath} → ${fd.newPath}`
                                                                        : `${fd.oldPath} → ${fd.newPath}`
                                                                )
                                                        )
                                                )
                                        )}
                                </span>
                            </Stack>
                            <span style={{color: tokens.colorNeutralForeground4}}>
                                {fd.type === 'add' && <>New file</>}
                                {fd.type === 'delete' && <>Deleted</>}
                            </span>
                        </Stack>,
                        children: <>
                            <Diff viewType="split" diffType={fd.type} hunks={fd.hunks}>
                                {hunks => hunks.map(hunk =>
                                    <Hunk key={hunk.content} hunk={hunk}/>)}
                            </Diff>
                        </>
                    });
                })}
            />
        </>}
    </>
};

