import * as React from "react";
import {DatasetRevisionsListEntry} from "meteoio-platform-client";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Text, tokens} from "@fluentui/react-components";
import {RelativeTimeText} from "meteoio-ui/src/components/RelativeTimeText";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {useDsRevisionMsgForm} from "./useDsRevisionMsgForm";
import {Button} from 'meteoio-ui/src/components/Button';
import {useDsRevActions} from "./useDsRevActions";
import {useScreenSize} from "meteoio-ui/src/hooks/useScreenSize";
import {Drawer} from "antd";
import {ThemeProvider} from "meteoio-ui/src/providers/ThemeProvider";

export const DsRevisionDetails: React.FC<{
    pending?: boolean;
    row: DatasetRevisionsListEntry;
    previousRevision?: DatasetRevisionsListEntry;
    isLatest?: boolean;
}> = props => {
    const {row} = props;
    const revisionMsgForm = useDsRevisionMsgForm({withConfirmBtn: props.pending});
    const dsRevActions = useDsRevActions(row, props.isLatest, props.previousRevision);
    const {isMobile} = useScreenSize()

    return <>
        <div style={{
            borderBottom: `1px solid ${tokens.colorNeutralStroke2}`,
            padding: '1em 24px',
            position: 'sticky',
            top: 0,
            zIndex: 1,
            backgroundColor: tokens.colorNeutralBackground1
        }}>
            <PageSectionTitle>
                {props.pending ? 'Pending revision' : `Version ${row.number}`}
            </PageSectionTitle>
        </div>
        <div style={{padding: '1em 24px'}}>
            <Stack rowGap="XXXL">
                
                <Stack rowGap="M">
                    {row.title && <div>
                        <Text block size={300} style={{fontWeight: '600', marginBottom: '0.5em'}}>
                            {row.title}
                        </Text>
                    </div>}
                    
                    {row.message && <div>
                        <Text block size={200} style={{color: tokens.colorNeutralForeground1, whiteSpace: 'pre-wrap'}}>
                            {row.message}
                        </Text>
                    </div>}
                    
                    {(!row.title && !row.message) && <Text block size={200} style={{color: tokens.colorNeutralForeground3, fontStyle: 'italic'}}>
                        No title or message set for this revision.
                    </Text>}

                    {props.pending && <Button
                        label={props.pending ? 'Edit and Confirm...' : 'Edit revision message'}
                        appearance={props.pending ? "primary" : 'subtle'}
                        onClick={() => revisionMsgForm.open(row)}
                    />}
                </Stack>

                <Stack horizontal wrap={isMobile} columnGap="XXXL">
                
                    <Stack rowGap="XXS" width="fit-content">
                        <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Version number: {row.number === null ? '(to be assigned)' : row.number}
                        </Text>
                        {row.created_at && <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Created: <RelativeTimeText date={row.created_at}/>
                        </Text>}
                        {row.confirmed_at && <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Confirmed: <RelativeTimeText date={row.confirmed_at}/>
                        </Text>}
                        {row.confirmed_at && row.updated_at && row.updated_at != row.confirmed_at &&
                            <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                                Updated: <RelativeTimeText date={row.updated_at}/>
                            </Text>}
                        {row.emptied_at && <Text block size={200} style={{color: tokens.colorStatusDangerForeground1, fontWeight: 'bold'}}>
                            Contents cleared: <RelativeTimeText date={row.emptied_at}/>
                            <br />
                            <Text size={100} style={{fontWeight: 'normal'}}>
                                You won't be able to restore or compare this revision.
                            </Text>
                        </Text>}
                        <div>&nbsp;</div>
                    </Stack>

                    <div style={{display: 'flex', flexFlow: 'column', flexGrow: 1}}>
                        {!props.pending && <Button
                            label="Edit revision message"
                            appearance="subtle"
                            onClick={() => revisionMsgForm.open(row)}
                        />}
                        {dsRevActions?.diffOptions?.map?.((opt, i) => <Button
                            key={i}
                            {...opt.buttonProps}
                            appearance="subtle"
                        />)}
                        {dsRevActions?.otherActions?.map?.((oa, i) => <Button
                            key={i}
                            {...oa.buttonProps}
                            appearance="subtle"
                        />)}
                    </div>
                </Stack>
            </Stack>
            
            {dsRevActions.renderNode}
        </div>
        
        {revisionMsgForm.isOpen && <Drawer
            open={revisionMsgForm.isOpen}
            size="large"
            placement="right"
            onClose={() => revisionMsgForm.close()}
            title={<ThemeProvider>
                <PageSectionTitle>{props.pending ? 'Edit and Confirm Revision' : `Edit Version ${row.number}`}</PageSectionTitle>
            </ThemeProvider>}
            styles={{
                body: {padding: '1em 24px'},
            }}
        >
            <ThemeProvider>
                <Stack rowGap="XXXL">
                    {revisionMsgForm.formNode}
                    <Stack rowGap="XXS">
                        <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Version number: {row.number === null ? '(to be assigned)' : row.number}
                        </Text>
                        {row.created_at && <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Created: <RelativeTimeText date={row.created_at}/>
                        </Text>}
                        {row.confirmed_at && <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                            Confirmed: <RelativeTimeText date={row.confirmed_at}/>
                        </Text>}
                        {row.confirmed_at && row.updated_at && row.updated_at != row.confirmed_at &&
                            <Text block size={200} style={{color: tokens.colorNeutralForeground3}}>
                                Updated: <RelativeTimeText date={row.updated_at}/>
                            </Text>}
                        {row.emptied_at && <Text block size={200} style={{color: tokens.colorStatusDangerForeground1, fontWeight: 'bold'}}>
                            Contents cleared: <RelativeTimeText date={row.emptied_at}/>
                            <br />
                            <Text size={100} style={{fontWeight: 'normal'}}>
                                You won't be able to restore or compare this revision.
                            </Text>
                        </Text>}
                    </Stack>
                </Stack>
            </ThemeProvider>
        </Drawer>}
    </>
};