import * as React from 'react'
import {useState} from 'react'
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Button} from "meteoio-ui/src/components/Button";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {
    DatasetRevisionsListEntry,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdRestoreFromOtherTxIdRestoreConfig
} from "meteoio-platform-client";
import {DsRevisionDiff} from "./DsRevisionDiff";
import {useParams} from "react-router-dom";
import {useDatasetConfigTx} from "../../useDatasetConfigTx";
import {useDialogs} from "meteoio-ui/src/hooks/useDialogs";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {tokens} from "@fluentui/react-components";
import {useQueryClient} from "../../../_common/backend";
import {Modal} from "meteoio-ui/src/components/Modal";
import {getMaybeErrorMessageString} from "meteoio-ui/src/components/MaybeErrorAlert";

/** Modal that allows to restore a revision after reviewing a preview of the changes */
export const DsRevRestorer: React.FC<{
    restoreRev: DatasetRevisionsListEntry
    onDismiss: () => void
}> = props => {
    const {datasetId} = useParams();
    const [workingTxId, {isHead, beginConfigTx}] = useDatasetConfigTx(datasetId);
    const {message, confirm, strong_modal_info} = useDialogs()
    const queryClient = useQueryClient()
    const [showPreview, setShowPreview] = useState<boolean>(false)
    const [closable, setClosable] = useState<boolean>(true)

    if (workingTxId === props.restoreRev.id) {
        // Caller should prevent this situation.
        return null
    }

    return <>
        <Modal
            title="Dataset Configurartion Restore"
            footer={null}
            onCancel={closable ? props.onDismiss : () => {}}
            closable={closable}
            open
            width={showPreview ? 800 : undefined}
        >
            <Stack rowGap="L">
                <div>
                    Are you sure you want to restore the configuration from Version {props.restoreRev.number}?
                    <br />
                    {isHead && <span>A new revision will be created.</span>}
                    {!isHead && <span>If you have any pending changes, these will be discarded.</span>}
                </div>
                <Stack horizontal justifyContent="end">
                    {!showPreview && <Button
                        appearance="subtle"
                        onClick={() => setShowPreview(true)}
                        label="See preview"
                    />}
                    <Button
                        appearance="secondary"
                        onClick={props.onDismiss}
                        label="Cancel"
                    />
                    <AsyncActionButton
                        appearance="primary"
                        onClick={async () => {
                            setClosable(false)
                            try {
                                if (!await confirm('Are you sure you want to proceed?', {okType: 'danger', okText: 'Yes, restore now'})) {
                                    return
                                }
                                let txId = workingTxId;
                                if (isHead) {
                                    txId = await beginConfigTx()
                                }
                                await fetchInternalDatasetsDatasetIdConfigTxsTxIdRestoreFromOtherTxIdRestoreConfig({
                                    pathParams: {
                                        datasetId,
                                        txId,
                                        otherTxId: props.restoreRev.id,
                                    }
                                })
                                await queryClient.invalidateQueries()
                                message.success(`${isHead ? 'Started revision with configuration restored' : 'Restored configuration'} from Version ${props.restoreRev.number}.`)
                                props.onDismiss()
                            }
                            catch (e) {
                                strong_modal_info(getMaybeErrorMessageString(e))
                            }
                            finally {
                                setClosable(true)
                            }
                        }}
                        label="Restore"
                    />
                </Stack>
                {showPreview && <>
                    <div style={{borderTop: `1px solid ${tokens.colorNeutralStroke2}`, height: 0}}>&nbsp;</div>
                    <Stack rowGap="S">
                        <PageSectionTitle>Preview of the operation</PageSectionTitle>
                        <DsRevisionDiff
                            leftTxId={workingTxId}
                            leftLabel="Current version"
                            rightTxId={props.restoreRev.id}
                            rightLabel={`Restoring version ${props.restoreRev.number}`}
                        />
                    </Stack>
                </>}
            </Stack>
        </Modal>
    </>
}