import * as React from 'react'
import {useState} from 'react'
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Button} from "meteoio-ui/src/components/Button";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {
    DatasetRevisionsListEntry,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdClearContentsClearConfigContents
} from "meteoio-platform-client";
import {useParams} from "react-router-dom";
import {useDatasetConfigTx} from "../../useDatasetConfigTx";
import {useDialogs} from "meteoio-ui/src/hooks/useDialogs";
import {useQueryClient} from "../../../_common/backend";
import {Modal} from "meteoio-ui/src/components/Modal";
import {getMaybeErrorMessageString} from "meteoio-ui/src/components/MaybeErrorAlert";

export const DsRevClearContents: React.FC<{
    clearRev: DatasetRevisionsListEntry
    onDismiss: () => void
}> = props => {
    const {datasetId} = useParams();
    const [workingTxId] = useDatasetConfigTx(datasetId);
    const {message, confirm, strong_modal_info} = useDialogs()
    const queryClient = useQueryClient()
    const [closable, setClosable] = useState<boolean>(true)

    if (workingTxId === props.clearRev.id) {
        // Caller should prevent this situation.
        return null
    }

    return <>
        <Modal
            title="Dataset configurartion contents clear"
            footer={null}
            onCancel={closable ? props.onDismiss : () => {}}
            closable={closable}
            open
        >
            <Stack rowGap="L">
                <div>
                    Are you sure you want to clear the Version {props.clearRev.number}?
                    <br />
                    The revision number, title and message will be preserved, but the revision record will be emptied from the configuration data.
                </div>
                <Stack horizontal justifyContent="end">
                    <Button
                        appearance="secondary"
                        onClick={props.onDismiss}
                        disabled={!closable}
                        label="Go back"
                    />
                    <AsyncActionButton
                        appearance="primary"
                        onClick={async () => {
                            setClosable(false)
                            try {
                                message.warning(`You are going to permanently clear this configuration version...`)
                                await new Promise(resolve => setTimeout(resolve, 2000))
                                if (!await confirm('Are you sure you want to proceed?', {okType: 'danger', okText: 'Yes, clear now'})) {
                                    return
                                }
                                await fetchInternalDatasetsDatasetIdConfigTxsTxIdClearContentsClearConfigContents({
                                    pathParams: {
                                        datasetId,
                                        txId: props.clearRev.id,
                                    }
                                })
                                await queryClient.invalidateQueries()
                                message.success(`Cleared contents of Version ${props.clearRev.number}.`)
                                props.onDismiss()
                            }
                            catch (e) {
                                strong_modal_info(getMaybeErrorMessageString(e))
                            }
                            finally {
                                setClosable(true)
                            }
                        }}
                        label="Clear"
                    />
                </Stack>
            </Stack>
        </Modal>
    </>
}