import * as React from 'react';
import {useState, useMemo} from 'react';
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {Modal} from "meteoio-ui/src/components/Modal";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Button} from "meteoio-ui/src/components/Button";
import {ArrowClockwiseFilled, WarningFilled} from '@fluentui/react-icons';
import {useParams} from "react-router-dom";
import {FsTreeSnapshotBrowser} from "meteoio-ui/src/components/FsTreeSnapshotBrowser";
import {useInternalDatasetsDatasetIdFsSnapshotTakeTakeFsSnapshot, useInternalDatasetsDatasetIdFsSnapshotCachedGetCachedFsSnapshot} from "meteoio-platform-client";
import {SimpleProperties, SimplePropertiesSchema} from "meteoio-ui/src/components/SimpleProperties";
import {useDatasetConfigTx} from "./useDatasetConfigTx";
import type * as Schemas from "meteoio-platform-client/generated/backendSchemas";
import {Alert} from "meteoio-ui/src/components/Alert";
import {PiTreeView} from "react-icons/pi";
import {RelativeTimeText} from "meteoio-ui/src/components/RelativeTimeText";

// NOTE: For dataset tree snapshotting including FUSE mounts [e85a0711-e823-493b-87d6-03cdbaae9642]

// Cache staleness thresholds (in milliseconds)
const CACHE_STALE_THRESHOLD = 10 * 60 * 1000;
const CACHE_VERY_STALE_THRESHOLD = 24 * 60 * 60 * 1000;

// Schema for the configuration properties
const configSchema: SimplePropertiesSchema = {
    // target_path: {
    //     title: 'Target Path',
    //     type: 'line',
    //     required: true
    // },
    soft_max_entries: {
        title: 'Soft Max Entries',
        type: 'int',
        min: 100,
        max: 10000,
        required: true
    },
    hard_max_entries: {
        title: 'Hard Max Entries',
        type: 'int',
        min: 100,
        max: 10000,
        required: true
    },
    soft_timeout: {
        title: 'Soft Timeout (seconds)',
        type: 'int',
        min: 1,
        max: 60,
        required: false
    },
    hard_timeout: {
        title: 'Hard Timeout (seconds)',
        type: 'int',
        min: 1,
        max: 60,
        required: false
    }
};

export const DatasetSnapshotButton: React.FC = () => {
    const {datasetId} = useParams();
    const [txId, {isHead}] = useDatasetConfigTx(datasetId)
    const [isConfigModalOpen, setIsConfigModalOpen] = useState<boolean>(false);
    const [isSnapshotModalOpen, setIsSnapshotModalOpen] = useState<boolean>(false);

    // Snapshot configuration state
    const [config, setConfig] = useState<Schemas.FsSnapshotJobParams>({
        soft_max_entries: 1000,
        hard_max_entries: 10000,
        soft_timeout: 15,
        hard_timeout: 60,
    });

    // Use React Query to fetch cached snapshot
    const cachedSnapshot = useInternalDatasetsDatasetIdFsSnapshotCachedGetCachedFsSnapshot({
        pathParams: { datasetId },
        queryParams: { tx_id: txId },
    }, {
        enabled: !!datasetId && !!txId,
    });

    // Use React Query mutation to generate snapshot
    const generateSnapshotMutation = useInternalDatasetsDatasetIdFsSnapshotTakeTakeFsSnapshot({
        onSuccess: () => {
            // Refetch the cached snapshot query
            cachedSnapshot.refetch().then();
            setIsConfigModalOpen(false)
            setIsSnapshotModalOpen(true)
        },
        onError: (error) => {
            console.error('Failed to generate snapshot:', error)
        }
    });

    const handleGenerateSnapshot = async () => {
        await new Promise((resolve, reject) => generateSnapshotMutation.mutate({
            pathParams: { datasetId },
            queryParams: { tx_id: txId },
            body: config,
        }, {
            onError(error) {
                reject(error)
            },
            onSuccess() {
                resolve(undefined);
            },
        }));
    };

    // Memoize cache age info to avoid recalculating on each render
    const cacheAgeInfo = useMemo(() => {
        // Helper function to determine cache staleness
        const getCacheAgeInfo = (cachedAt: string) => {
            const cacheDate = new Date(cachedAt);
            const now = new Date();
            const ageMs = now.getTime() - cacheDate.getTime();

            return {
                isStale: ageMs > CACHE_STALE_THRESHOLD,
                isVeryStale: ageMs > CACHE_VERY_STALE_THRESHOLD,
            };
        };
        return cachedSnapshot.data ? getCacheAgeInfo(cachedSnapshot.data.cached_at) : null;
    }, [cachedSnapshot.data?.cached_at]);

    return <>
        <Button
            // @ts-ignore
            icon={<PiTreeView/>}
            label="Snapshot files tree"
            disabled={cachedSnapshot.isLoading}
            onClick={() => {
                if (cachedSnapshot.data) {
                    setIsSnapshotModalOpen(true)
                } else {
                    setIsConfigModalOpen(true)
                }
            }}
        />

        {/* Configuration Modal */}
        <Modal
            title="Create Filesystem Snapshot"
            open={isConfigModalOpen}
            onCancel={() => setIsConfigModalOpen(false)}
            footer={null}
        >
            <Stack rowGap="M">
                <span>
                    Create a snapshot of the dataset's files tree, including external file systems.
                    It will allow you to browse the directories as prepared for job execution.
                </span>
                {!isHead && <Alert level="warning">
                    The external file systems configuration will be from your pending revision.
                </Alert>}

                <Stack rowGap="XXS">
                    <SimpleProperties
                        value={config}
                        onChange={setConfig}
                        schema={configSchema}
                    />
                </Stack>

                <Alert level="low_info">
                    The operation will be queued.
                    Wait time is excluded from the above timeout setting.
                    If remote file systems are mounted, network latency may slow down the scan.
                </Alert>

                <Stack horizontal justifyContent="end" columnGap="M">
                    <Button
                        label="Cancel"
                        onClick={() => setIsConfigModalOpen(false)}
                    />
                    <AsyncActionButton
                        label="Create Snapshot"
                        appearance="primary"
                        // isAdditionallyPending={generateSnapshotMutation.isLoading}
                        onClick={handleGenerateSnapshot}
                    />
                </Stack>
            </Stack>
        </Modal>

        {/* Snapshot Viewer Modal */}
        <Modal
            title="Filesystem Snapshot"
            open={!!cachedSnapshot.data && isSnapshotModalOpen}
            onCancel={() => setIsSnapshotModalOpen(false)}
            width="90vw"
            footer={null}
        >
            {cachedSnapshot.data && (
                <div style={{minHeight: 'calc(95vh - 200px)'}}>
                    <span>
                        This is a snapshot of the dataset's filesystem, including external file systems.
                        It allows you to browse the directories as prepared for job execution.
                    </span>
                    <Stack horizontal justifyContent="space-between">
                        <div style={{
                            opacity: cacheAgeInfo?.isStale ? 1 : 0.7,
                            color: cacheAgeInfo?.isVeryStale ? '#d13438' : undefined
                        }}>
                            <Stack horizontal columnGap="XS" alignItems="center">
                                {cacheAgeInfo?.isStale && (
                                    <WarningFilled style={{ fontSize: '16px', color: '#d13438' }} />
                                )}
                                <span>
                                    Snapshot date: <RelativeTimeText date={new Date(cachedSnapshot.data.cached_at)}/>
                                </span>
                            </Stack>
                        </div>
                        <Button
                            label="Retake snapshot"
                            size="small"
                            appearance={cacheAgeInfo?.isStale ? "primary" : undefined}
                            icon={<ArrowClockwiseFilled title="Refresh" style={{scale: '0.8'}}/>}
                            onClick={() => {
                                setIsSnapshotModalOpen(false)
                                setIsConfigModalOpen(true)
                            }}
                        />
                    </Stack>
                    <br/>
                    <FsTreeSnapshotBrowser
                        tree={cachedSnapshot.data.tree}
                        rootLabel={`Dataset root`}
                    />
                </div>
            )}
        </Modal>
    </>;
};