import * as React from 'react'
import {SimpleProperties} from "meteoio-ui/src/components/SimpleProperties";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {
    fetchInternalDatasetsDatasetIdConfigTxsBeginConfigWriteTx,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseGetFuse,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseSetFuse,
} from "meteoio-platform-client";
import {useDatasetConfigTx} from "./useDatasetConfigTx";
import {useNavigate, useParams} from "react-router-dom";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {getMaybeErrorMessageString, MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {AddRegular, BinRecycleRegular, EditRegular} from "@fluentui/react-icons";
import {useDatasetConfigTxDraftPusher} from "./useDatasetConfigTxDraftPusher";
import {App, Collapse, Empty} from "antd";
import {Badge, Menu, MenuButton, MenuItem, MenuList, MenuPopover, MenuTrigger} from "@fluentui/react-components";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {address_book} from "../address_book";
import {COMMON_FIELDS_SCHEMA, FuseConfig, FuseConfigElement, SCHEMAS} from "./DatasetFuseConfigFormSchemas";
import {Modal} from "meteoio-ui/src/components/Modal";
import {SetStateAction, useState} from "react";


export const DatasetFuseSettings: React.FC = () => {
    return <>
        <Stack rowGap="XL" width={500} maxWidth="90%" center>
            <FuseConfigEditor/>
            <br/>
            <br/>
        </Stack>
    </>
}

const FuseConfigEditor: React.FC = () => {
    const {datasetId, fuseConfKey} = useParams()
    const [txId, {isHead, setConfigTxId}] = useDatasetConfigTx(datasetId)
    const navigate = useNavigate()
    const {modal} = App.useApp()

    const [draft, setDraft, {error, isLoading, reload}] = useDatasetConfigTxDraftPusher<FuseConfig>(
        `8e072dc5-9c56-403b-8aa8-17e36082ffc7-ds-${datasetId}-tx-${txId}`,
        isHead ? undefined : async (draft) => {
            try {
                await fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseSetFuse({
                    pathParams: {datasetId, txId},
                    body: draft
                })
            }
            catch (e) {
                modal.error({
                    content: getMaybeErrorMessageString(e)
                })
            }
        },
        async () => {
            return await fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseGetFuse({
                pathParams: {
                    datasetId,
                    txId,
                },
            })
        })

    // NOTE: without editing modal, there's an issue with cache invalidation after tx discard [1ec14e60-f61a-40b4-a4b2-77209cac2169]
    //              DatasetHeadingSettings might also be affected.
    const [editing, setEditing] = useState<{key: string, conf: FuseConfigElement } | null>(null)

    // TODO: show warning if duplicate local_mount_sub_path

    return <>
        {/*<PageTitle>External file systems</PageTitle>*/}
        <Stack horizontal justifyContent="space-between">
            <PageSectionTitle>
                Configured mounts&nbsp;
                {isLoading ? <></> : <>({Object.keys(draft?.mounts??{}).length})</>}
            </PageSectionTitle>
            {isHead ? <>
                <AsyncActionButton
                    icon={<EditRegular/>}
                    label="Start editing..."
                    onClick={async () => {
                        const res = await fetchInternalDatasetsDatasetIdConfigTxsBeginConfigWriteTx({
                            pathParams: {
                                datasetId,
                            }
                        })
                        setConfigTxId(res.id)
                    }}
                />
            </> : <>
                <Menu>
                    <MenuTrigger disableButtonEnhancement>
                        <MenuButton icon={<AddRegular/>} appearance="primary">Add...</MenuButton>
                    </MenuTrigger>
                    <MenuPopover>
                        <MenuList>
                            {SCHEMAS.map(s => <MenuItem
                                    key={s.option.value}
                                    title={s.option.value}
                                    onClick={() => {
                                        const key = crypto.randomUUID()
                                        setEditing({
                                            key,
                                            conf: {
                                                driver: s.option.value
                                            } as FuseConfigElement
                                        })
                                        navigate(address_book.datasets.settings.fuse_view_one(datasetId, key))
                                    }}
                                >
                                    {s.option.label}
                                </MenuItem>)}
                        </MenuList>
                    </MenuPopover>
                </Menu>
            </>}
        </Stack>
        <MaybeErrorAlert error={error}/>
        <Collapse
            activeKey={fuseConfKey}
            onChange={key => navigate(address_book.datasets.settings.fuse_view_one(datasetId, key))}
            size="small"
            accordion
            items={Object.entries(draft?.mounts??{}).map(([key, conf]) => {

                const removeAction: SetStateAction<FuseConfig> = _conf => ({mounts: Object.fromEntries(
                        Object.entries(_conf.mounts).filter(([k, v]) => k !== key))})

                const schema_entry = SCHEMAS.find(s => s.option.value === conf.driver)
                return {
                    key,
                    label: <Stack horizontal justifyContent="space-between">
                        <div style={{overflow: 'hidden', whiteSpace: 'nowrap', wordBreak: 'break-all', maxWidth: '300px', textOverflow: 'ellipsis'}}>
                            <Badge size="small" appearance="tint">{schema_entry.option.label}</Badge>
                            &nbsp;
                            &nbsp;
                            <code>
                                {conf.local_mount_sub_path}
                            </code>
                        </div>
                        {!isHead && <Stack horizontal>
                            <AsyncActionButton
                                appearance="secondary"
                                size="small"
                                icon={<EditRegular/>}
                                label="Edit"
                                onClick={async (ev) => {
                                    ev.stopPropagation()
                                    setEditing({key, conf})
                                }}
                            />
                            <AsyncActionButton
                                appearance="secondary"
                                danger
                                size="small"
                                icon={<BinRecycleRegular/>}
                                onClick={async (ev) => {
                                    ev.stopPropagation()
                                    if (!await modal.confirm({
                                        title: 'Are you sure you want to remove this mount?'
                                    })) {
                                        return
                                    }
                                    setDraft(removeAction)
                                }}
                            />
                        </Stack>}
                    </Stack>,
                    children: <>
                        <SimpleProperties
                            schema={COMMON_FIELDS_SCHEMA}
                            value={conf}
                            // onChange={v => setAll(v)}
                            disabled // ={isHead}
                        />
                        <br />
                        {schema_entry && <SimpleProperties
                            schema={schema_entry.schema}
                            value={conf}
                            // onChange={v => setAll(v)}
                            disabled // ={isHead}
                        />}
                    </>
                }
            })}
        />
        {isLoading ? <Spinner tall/> : <>
            {Object.entries(draft?.mounts??{})?.length <= 0 && <Empty description="No external file systems"/>}
        </>}

        {editing && <Modal
            open={!!editing}
            onCancel={() => {}}
            footer={null}
            closable={false}
        >
            <PageSectionTitle>
                External file system mount settings
                &nbsp;
                <Badge appearance="tint">{SCHEMAS.find(s => s.option.value === editing.conf.driver)?.option?.label}</Badge>
            </PageSectionTitle>
            <br />
            <SimpleProperties
                schema={COMMON_FIELDS_SCHEMA}
                value={editing.conf}
                onChange={v => setEditing({key: editing.key, conf: v as FuseConfigElement})}
            />
            <br />
            <SimpleProperties
                schema={SCHEMAS.find(s => s.option.value === editing.conf.driver)?.schema ?? {}}
                value={editing.conf}
                onChange={v => setEditing({key: editing.key, conf: v as FuseConfigElement})}
            />
            <br />
            <Stack horizontal justifyContent="flex-end">
                <AsyncActionButton
                    appearance="subtle"
                    label="Delete..."
                    danger
                    icon={<BinRecycleRegular/>}
                    onClick={async () => {
                        if (!await modal.confirm({
                            title: 'Are you sure you want to remove this mount?'
                        })) {
                            return
                        }
                        const removeAction: SetStateAction<FuseConfig> = _conf => ({mounts: Object.fromEntries(
                                Object.entries(_conf.mounts).filter(([k, v]) => k !== editing.key))})
                        const newDraft = removeAction(draft)
                        await fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseSetFuse({
                            pathParams: {datasetId, txId},
                            body: newDraft
                        })
                        setDraft(removeAction)
                        setEditing(null)
                    }}
                />
                <AsyncActionButton
                    label="Cancel"
                    onClick={async () => {
                        if (!await modal.confirm({
                            title: 'Are you sure you want to discard your pending edits?'
                        })) {
                            return
                        }
                        setEditing(null)
                    }}
                />
                <AsyncActionButton
                    appearance="primary"
                    label="Save"
                    onClick={async () => {
                        const newDraft: FuseConfig = {
                            mounts: {
                                ...(draft?.mounts ?? {}),
                                [editing.key]: editing.conf,
                            }
                        }
                        try {
                            await fetchInternalDatasetsDatasetIdConfigTxsTxIdFuseSetFuse({
                                pathParams: {datasetId, txId},
                                body: newDraft
                            })
                            await reload()  // This, instead of setDraft, allows default values to be loaded from the server.
                            setEditing(null)
                            // NOTE: given that this modal can be opened with edit buttons without opening the collapse, do not navigate(address_book.datasets.settings.fuse_view_one(datasetId, editing.key))
                        }
                        catch (e) {
                            modal.error({
                                content: getMaybeErrorMessageString(e)
                            })
                        }
                    }}
                />
            </Stack>
        </Modal>}
    </>
}
