import * as React from 'react'
import {useCallback, useEffect, useState} from 'react'
import {CronEditor} from "meteoio-ui/src/components/CronEditor";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {DropdownField} from "meteoio-ui/src/components/DropdownField";
import {InfoBox} from "meteoio-ui/src/components/InfoBox";
import {Sticky} from "meteoio-ui/src/components/Sticky";
import {Collapse, Empty} from 'antd'
import {CronHumanFrag} from "meteoio-ui/src/components/CronHumanFrag";
import {Badge, Text} from '@fluentui/react-components'
import {AddRegular, BinRecycleRegular, EditRegular, PlayFilled} from "@fluentui/react-icons";
import {Link, useNavigate, useParams} from "react-router-dom";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {
    CronJob,
    fetchInternalDatasetsDatasetIdConfigTxsBeginConfigWriteTx,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsCreateCronJob,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdDeleteCronJob,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdReadCronJob,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdTriggerTriggerCronJob,
    fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdUpdateCronJob,
    TimeSeriesJobParams,
    useInternalDatasetsDatasetIdConfigTxsTxIdCronJobsListCronJobs,
    useInternalDatasetsDatasetIdConfigTxsTxIdInisListInis
} from "meteoio-platform-client";
import {useDatasetConfigTx} from "./useDatasetConfigTx";
import {TimeSeriesJobParamsEditor} from "../_common/TimeSeriesJobParamsEditor";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {useStateTransitionCallback} from "meteoio-ui/src/hooks/useStateTransitionEffect";
import {useDialogs} from "meteoio-ui/src/hooks/useDialogs";
import {address_book} from "../address_book";
import {EnvSettingsEditor} from "../_common/EnvSettingsEditor";
import {useDatasetConfigTxDraftPusher} from "./useDatasetConfigTxDraftPusher";
import {useQueryClient} from "../_common/backend";


export const DatasetCronTab: React.FC = () => {
    const {datasetId, cron_job_id} = useParams()
    const [txId, {isHead, setConfigTxId}] = useDatasetConfigTx(datasetId)
    const remoteCronJobs = useInternalDatasetsDatasetIdConfigTxsTxIdCronJobsListCronJobs({
        pathParams: {datasetId, txId}
    })

    const remoteInis = useInternalDatasetsDatasetIdConfigTxsTxIdInisListInis({
        pathParams: {datasetId, txId}
    })

    //const [openKey, setOpenKey] = useState<string | string[]>(undefined)
    const navigate = useNavigate()
    const openKey = cron_job_id
    const setOpenKey = useCallback((openKey: string) => navigate(address_book.datasets.cron(datasetId, openKey)), [datasetId])

    const [draftCronJobs, setDraftCronJobs] = useState<CronJob[] | undefined>()
    useStateTransitionCallback(txId, () => {
        setDraftCronJobs(undefined)
        remoteCronJobs.refetch().then()
    }, [])
    useEffect(() => {
        if (isHead || (draftCronJobs === undefined)) {
            setDraftCronJobs(remoteCronJobs.data)
        }
    }, [remoteCronJobs.data, draftCronJobs, isHead])

    const [pushError, setPushError] = useState<Error>()

    const dialogs = useDialogs()

    if (remoteCronJobs.isLoading || (draftCronJobs === undefined && !remoteCronJobs.error)) {
        return <Spinner tall/>
    }
    if (remoteCronJobs.error) {
        return <MaybeErrorAlert error={remoteCronJobs.error}/>
    }
    return <>
        <main>
            <Stack rowGap="L">
                <MaybeErrorAlert error={pushError}/>
                {draftCronJobs?.length > 0 && <Collapse
                    activeKey={openKey}
                    accordion
                    onChange={setOpenKey}
                    items={draftCronJobs?.map?.(job => ({
                        key: job.id,
                        label: <Stack horizontal justifyContent="space-between">
                            <Text block truncate>
                                <Badge appearance="tint">
                                    <code style={{fontSize: '0.8em'}}>{job.id?.slice?.(-8)}</code>
                                </Badge>
                                {/*<code style={{fontSize: '0.8em', color: tokens.colorBrandForegroundLinkPressed}}>{job.id?.slice?.(-8)}</code>*/}
                                &nbsp;
                                &nbsp;
                                <CronHumanFrag cron={job.cron}/>
                                {/*{WORKFLOWS.find(wf => wf.option.value === job.workflow)?.option?.label}*/}
                            </Text>
                            <Stack horizontal>
                                <AsyncActionButton
                                    appearance="secondary"
                                    size="small"
                                    icon={<PlayFilled color="green"/>}
                                    label="Run now"
                                    onClick={async () => {
                                        if (!await dialogs.confirm('Are you sure you want to run this cron job now?')) {
                                            return
                                        }
                                        const result = await fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdTriggerTriggerCronJob({
                                            pathParams: {
                                                datasetId,
                                                txId,
                                                jobId: job.id,
                                            }
                                        })
                                        navigate(address_book.jobs.view(result.id))
                                    }}
                                />
                                {!isHead && <AsyncActionButton
                                    appearance="secondary"
                                    danger
                                    size="small"
                                    icon={<BinRecycleRegular/>}
                                    onClick={async () => {
                                        if (!await dialogs.confirm('Are you sure you want to remove this cron job?')) {
                                            return
                                        }
                                        await fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdDeleteCronJob({
                                            pathParams: {
                                                datasetId,
                                                txId,
                                                jobId: job.id,
                                            }
                                        })
                                        await remoteCronJobs.refetch()
                                        setDraftCronJobs(draftCronJobs?.filter?.(j => j.id !== job.id))
                                    }}
                                />}
                            </Stack>
                        </Stack>,
                        children: <DatasetCronjobEditor
                            disabled={isHead}
                            value={job}
                            onChange={newJob => {
                                if (isHead) {
                                    return
                                }
                                setDraftCronJobs(draftCronJobs?.map?.(oldJob => oldJob.id === newJob.id ? newJob : oldJob) ?? [newJob])
                            }}
                        />,
                    }))}
                />}

                {draftCronJobs?.length <= 0 && <Empty description="No cron jobs"/>}

            </Stack>
            <br/>
        </main>
        <aside>
            <Sticky top={75}>
                <Stack>
                    {isHead && <AsyncActionButton
                        icon={<EditRegular/>}
                        label="Start editing..."
                        onClick={async () => {
                            // await new Promise(resolve => setTimeout(resolve, 300))
                            // throw new Error('Err')
                            const txIdDto = await fetchInternalDatasetsDatasetIdConfigTxsBeginConfigWriteTx({
                                pathParams: {datasetId}
                            })
                            setConfigTxId(txIdDto.id)
                        }}
                    />}
                    {!isHead && <AsyncActionButton
                        label="Create a new Cron Job"
                        appearance="primary"
                        icon={<AddRegular/>}
                        onClick={async () => {
                            const newJob: CronJob = {
                                cron: '*/10 * * * *',
                                type: 'meteoio_timeseries',
                                params: {
                                    ini: (remoteInis.data)?.[0],
                                    range: {
                                        duration_days: 30,
                                        end: 'NOW',
                                    },
                                    resolution_minutes: null,
                                },
                                // id: 'uncreated' + makeRandomUuid(),  // NOTE: old implementation required this, but was later fixed.
                            }
                            try {
                                const jobIdDto = await fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsCreateCronJob({
                                    pathParams: {
                                        datasetId,
                                        txId,
                                    },
                                    body: newJob,
                                })
                                // Store the id in the client list
                                newJob.id = jobIdDto.id
                                setDraftCronJobs(jobs => [...jobs, newJob])
                                setOpenKey(newJob.id)
                            } catch (e) {
                                console.trace(e)
                                setPushError(e)
                                setDraftCronJobs(undefined)
                                remoteCronJobs.refetch().then()
                            }
                        }}
                    />}
                </Stack>
                <br/>
                <InfoBox>
                    A cron job is a command or script that is scheduled to run at specific intervals or
                    fixed times (e.g., every hour, daily, weekly).
                    By using this page, users can schedule cron jobs to run.
                    The system reads this configuration periodically and triggers the execution of the
                    corresponding cron jobs based on their schedules.
                    Take a look at the Logs to monitor their execution.
                </InfoBox>
                <br/>
                <InfoBox noHead>
                    Environment variables have the following priorities (from highest to lowest):
                    <ol style={{marginBlock: '0.25em 0', paddingInlineStart: '1.75em'}}>
                        <li>
                            <Link to={address_book.datasets.tab(datasetId, 'ini')}>
                                INI configuration file
                            </Link>
                        </li>
                        <li>
                            <b>Cron Job</b>
                        </li>
                        <li>
                            <Link to={address_book.datasets.settings.job_env(datasetId)}>
                                Dataset
                            </Link>
                        </li>
                        <li>
                            <Link to={address_book.user.job_env}>
                                User profile
                            </Link>
                        </li>
                        <li>
                            <Link to={address_book.admin.job_env}>
                                System
                            </Link>
                        </li>
                        <li>
                            MeteoIO default value
                        </li>
                    </ol>
                </InfoBox>
            </Sticky>
        </aside>
    </>
}

export const DatasetCronjobEditor: React.FC<{
    disabled?: boolean
    value?: CronJob
    onChange?: (value: CronJob) => void
}> = props => {
    const {datasetId} = useParams()
    const [txId, {isHead}] = useDatasetConfigTx(datasetId)
    const remoteInis = useInternalDatasetsDatasetIdConfigTxsTxIdInisListInis({
        pathParams: {datasetId, txId}
    })

    const queryClient = useQueryClient()

    // const [isEnvProfileOpen, setIsEnvProfileOpen] = useState<boolean>(false)

    const [draft, setDraft, pushStatus] = useDatasetConfigTxDraftPusher(`f7d56092-cbcd-4cb4-9018-7d26e0c2b5c1-${datasetId}-${txId}-${props?.value?.id ?? 'nocron'}`, async (cronJobData) => {
        if (props.disabled || isHead) {  // redundant conditions in OR -- one is sufficient, but both mean the same, XOR would be unexpected.
            return
        }
        await fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdUpdateCronJob({
            pathParams: {
                datasetId, txId, jobId: props.value.id,
            },
            body: cronJobData,
        })
        await queryClient.invalidateQueries()  // see usage of remoteCronJobs list
    }, async () => {
        return await fetchInternalDatasetsDatasetIdConfigTxsTxIdCronJobsJobIdReadCronJob({
            pathParams: {
                datasetId, txId, jobId: props.value.id,
            }
        })
    })

    const shownValue = draft ?? props.value
    const onChange: (value: CronJob) => void = (data) => {
        setDraft(data)
        props.onChange(data)
    } //old: props.onChange

    return <>
        <Stack rowGap="L">
            <MaybeErrorAlert error={pushStatus.error}/>
            <CronEditor
                readOnly={props.disabled}
                label="Schedule"
                value={shownValue?.cron}
                onChange={value => {
                    onChange?.({...shownValue, cron: value})
                }}
            />
            <DropdownField
                label={"Workflow"}
                options={[{label: 'MeteoIO Timeseries', value: 'meteoio_timeseries'}]}
                value={shownValue.type}
                onChange={value => {
                    onChange?.({...shownValue, type: value})
                }}
            />
            <Stack rowGap="S">
                <TimeSeriesJobParamsEditor
                    iniFileNames={remoteInis.data}
                    inisError={remoteInis.error}
                    inisLoading={remoteInis.isLoading}
                    value={shownValue.params}
                    onChange={value => {
                        onChange?.({...shownValue, params: value as TimeSeriesJobParams})
                    }}
                />
            </Stack>
            {/*<AsyncActionButton*/}
            {/*    appearance="secondary"*/}
            {/*    label={props.disabled ? "View environment variables..." : "Edit environment variables..."}*/}
            {/*    onClick={async () => {*/}
            {/*        setIsEnvProfileOpen(true)*/}
            {/*    }}*/}
            {/*/>*/}
            <EnvSettingsEditor
                level="cron"
                disabled={isHead}
                useImmediateOnly
                immediateLoadedValue={shownValue?.job_env?.entries ?? []}
                onChangeImmediate={entries => onChange?.({...shownValue, job_env: {entries}})}
                compilingDatasetId={datasetId}
                compilingDatasetTxId={txId}
            />
        </Stack>
        {/*<Modal*/}
        {/*    open={isEnvProfileOpen}*/}
        {/*    footer={null}*/}
        {/*    closable={false}*/}
        {/*    maskClosable={false}*/}
        {/*    // keyboard={false}*/}
        {/*    // width={800}*/}
        {/*>*/}
        {/*    {isEnvProfileOpen && <EnvSettingsEditor*/}
        {/*        onStore={async (entries) => {*/}
        {/*            onChange?.({...shownValue, job_env: {entries}})*/}
        {/*        }}*/}
        {/*        immediateLoadedValue={shownValue?.job_env?.entries ?? []}*/}
        {/*        // onLoad={async () => {*/}
        {/*        //     return*/}
        {/*        // }}*/}
        {/*        onDiscard={() => setIsEnvProfileOpen(false)}*/}
        {/*    />}*/}
        {/*</Modal>*/}
    </>
}