import * as React from 'react'
import {useEffect, useMemo, useState} from 'react'
import {
    fetchAdminUsersEditUserIdCanCreateDatasetsResetCanCreateDatasets,
    fetchAdminUsersEditUserIdCanCreateDatasetsSetCanCreateDatasets,
    useAdminUsersListUsers,
    usePublicMeGetMe,
    UserOidReflection,
    UserReflection
} from "meteoio-platform-client";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {Column, SortableTable} from "meteoio-ui/src/components/SortableTable";
import {Pagination, Switch} from 'antd'
import {PageTitle} from "meteoio-ui/src/components/PageTitle";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Text} from "@fluentui/react-components";
import {ORCIDiD_IconVector} from "meteoio-ui/src/components/ORCIDiD_IconVector";
import {Link} from "react-router-dom";
import {ProgressBar} from "meteoio-ui/src/components/ProgressBar";
import {Button} from "meteoio-ui/src/components/Button";
import {ArrowClockwiseRegular} from "@fluentui/react-icons";
import {useQueryClient} from "../_common/backend";
import {RelativeTimeText} from "meteoio-ui/src/components/RelativeTimeText";
import {useNavigate} from "react-router-dom";

const COLUMNS: Column<UserReflection>[] = [
    {
        field: 'display_name',
        title: 'Name',
        onRenderCellContent: row => (
            <span title={row.full_name ? `Full name: ${row.full_name}` : undefined}>
                {row.display_name || row.id}
            </span>
        ),
    },
    {
        field: 'email',
        title: 'Email',
    },
    // {
    //     field: 'has_password',
    //     title: 'Has password',
    //     onRenderCellContent: row => row.has_password ? <>&#10003;</> : <>&#8709;</>,
    // },
    /*{
        title: 'Credentials',
        onRenderCellContent: row => <Stack horizontal columnGap="XXXL" wrap alignItems="center">
            {row.has_password && row.email && <>
                <Stack horizontal columnGap="XS" alignItems="center">
                    <KeyRegular fontSize="1.3em"/>
                    email/password
                </Stack>
            </>}
            {row.openid?.map?.(oid => ['https://sandbox.orcid.org/', 'https://orcid.org/'].includes(oid.issuer)
                ? <Stack horizontal columnGap="XS" alignItems="center">
                    <span style={{textAlign: 'right', fontSize: '1.35em'}}>
                        <ORCIDiD_IconVector/>
                    </span>
                    <Link href={`${oid.issuer}${oid.subject}`} style={{fontSize: '0.9em'}} target="_blank">
                        {oid.subject}
                    </Link>
                </Stack>
                : <span>{oid.issuer} {oid.subject}</span>)}
        </Stack>
    },*/
    {
        field: 'created_at',
        title: 'Created',
        onRenderCellContent: row => row.created_at && 
            <RelativeTimeText date={row.created_at}/>
    },
    /* {
        field: 'id',
        title: 'ID',
        onRenderCellContent: row => (
            <Link to={`/admin/users/${row.id}`}>
                <code
                    style={{
                        lineHeight: '0.9em',
                        fontSize: '0.8em',
                        display: 'block',
                        overflow: 'hidden',
                        textOverflow: 'ellipsis',
                    }}
                >
                    {row.id}
                </code>
            </Link>
        ),
    },*/
]

/*
const OID_COLUMNS: Column<UserOidReflection>[] = [
    {
        field: 'issuer',
        title: 'OpenID issuer',
        onRenderCellContent: row => ['https://sandbox.orcid.org/', 'https://orcid.org/'].includes(row.issuer)
            ? <Stack horizontal columnGap="S" alignItems="center">
                    <span style={{textAlign: 'right', fontSize: '1.4em'}}>
                        <ORCIDiD_IconVector/>
                    </span>
                <span>{row.issuer}</span>
            </Stack>
            : row.issuer
    },
    {
        field: 'subject',
        onRenderCellContent: row => ['https://sandbox.orcid.org/', 'https://orcid.org/'].includes(row.issuer)
            ? <Link href={`${row.issuer}${row.subject}`} target="_blank">
                {row.subject}
            </Link>
            : row.subject
    },
]
*/

export const AdminPage_Users: React.FC = () => {
    const [limit, setLimit] = useState<number>(10);
    const [offset, setOffset] = useState<number>(0);
    const { data, isLoading, error, isFetching, refetch } = useAdminUsersListUsers({
        queryParams: { offset, limit },
    });
    const navigate = useNavigate()

    if (error) {
        return <MaybeErrorAlert error={error} />;
    }
    if (isLoading) {
        return <Spinner tall />;
    }

    return (
        <Stack rowGap="XL">
            <Stack horizontal columnGap="L">
                <PageTitle>Users</PageTitle>
                <Button
                    appearance="subtle"
                    icon={<ArrowClockwiseRegular />}
                    onClick={() => refetch().then()}
                />
            </Stack>
            <Text size={300} style={{ color: 'var(--colorNeutralForeground2)' }}>
                Total: {data.total} users
            </Text>
            <Stack rowGap="S">
                <ProgressBar hide={!isFetching} />
                <SortableTable
                    columns={COLUMNS}
                    rows={data.items ?? []}
                    missingPlaceholder={<span style={{ opacity: 0.5 }}>&#823;</span>}
                    onRowClick={row => {
                        navigate(`/admin/users/${row.id}`)
                    }}
                />
            </Stack>
            <Pagination
                current={1 + Math.floor(offset / limit)}
                total={data.total}
                pageSize={limit}
                responsive
                hideOnSinglePage
                showSizeChanger
                onChange={(page, pageSize) => {
                    setOffset((page - 1) * pageSize);
                    setLimit(pageSize);
                    document?.body?.scrollTo?.({ top: 0, behavior: 'smooth' });
                }}
            />
        </Stack>
    );
};

export const CanCreateDatasetsCtrl: React.FC<{
    user: UserReflection
}> = props => {
    const [can_create_datasets, _set_can_create_datasets] = useState<boolean>(() => !!props.user.can_create_datasets)
    const [isPending, setIsPending] = useState<boolean>(false)
    const [error, setError] = useState<Error | undefined>()

    const {data: my_user_data} = usePublicMeGetMe({})
    const queryClient = useQueryClient()

    useEffect(() => {
        _set_can_create_datasets(!!props.user.can_create_datasets)
    }, [props.user])

    return <>
        <Stack horizontal>
            <span>Can create datasets?</span>
            <Switch
                checked={can_create_datasets}
                checkedChildren={"Yes"}
                unCheckedChildren={"No"}
                loading={isPending}
                onChange={async checked => {
                    if (isPending) {
                        return
                    }
                    setIsPending(true)
                    setError(undefined)
                    try {
                        if (checked) {
                            await fetchAdminUsersEditUserIdCanCreateDatasetsSetCanCreateDatasets({
                                pathParams: {
                                    editUserId: props.user.id,
                                }
                            })
                        } else {
                            await fetchAdminUsersEditUserIdCanCreateDatasetsResetCanCreateDatasets({
                                pathParams: {
                                    editUserId: props.user.id,
                                }
                            })
                        }
                        _set_can_create_datasets(checked)
                    }
                    catch (e) {
                        setError(e)
                    }
                    finally {
                        await new Promise(resolve => setTimeout(resolve, 500))  // better visual feedback
                        setIsPending(false)
                        if (my_user_data?.id === props.user.id) {
                            queryClient.clear()
                        }
                    }
                }}
            />
            <MaybeErrorAlert error={error}/>
        </Stack>
    </>
}