import * as React from 'react'
import {useState} from 'react'
import {DatasetListEntry, useAdminDatasetsListDatasets,} from "meteoio-platform-client";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {PageTitle} from "meteoio-ui/src/components/PageTitle";
import {Column, SortableTable} from "meteoio-ui/src/components/SortableTable";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Pagination} from "antd";
import {Button} from "meteoio-ui/src/components/Button";
import {Text} from "@fluentui/react-components";
import {ArrowClockwiseRegular, bundleIcon, EyeFilled, EyeRegular} from '@fluentui/react-icons'
import {ProgressBar} from "meteoio-ui/src/components/ProgressBar";
import {useHref, useNavigate} from "react-router-dom";
import {MenuItemLink, MenuList} from "@fluentui/react-components";
import {RepositoryAccessPill} from "meteoio-ui/src/components/RepositoryAccessPill";
import {RelativeTimeText} from "meteoio-ui/src/components/RelativeTimeText";

const ViewIcon = bundleIcon(EyeFilled, EyeRegular)

const COLUMNS: Column<DatasetListEntry>[] = [
    {
        field: 'owner_user_id',
        title: 'Owner',
        onRenderCellContent: row => <code style={{
            lineHeight: '0.9em',
            fontSize: '0.8em',
            display: 'block',
            overflow: 'hidden',
            textOverflow: 'ellipsis'
        }}>
            {row.owner_user_id}
        </code>,
    },
    {
        field: 'id',
        title: 'Dataset ID',
        onRenderCellContent: row => <code style={{
            lineHeight: '0.9em',
            fontSize: '0.8em',
            display: 'block',
            overflow: 'hidden',
            textOverflow: 'ellipsis'
        }}>
            {row.id}
        </code>,
    },
    {
        title: 'Title',
        onRenderCellContent: row => row.heading && row.heading.title,
    },
    {
        field: 'access',
        title: 'Access',
        onRenderCellContent: row => <RepositoryAccessPill access={row.access}/>,
    },
    {
        field: 'created_at',
        title: 'Created',
        onRenderCellContent: row => row.created_at && 
            <RelativeTimeText date={row.created_at}/>
    },
]

const DatasetMenu: React.FC<{ row: DatasetListEntry }> = ({ row }) => {
    const datasetHref = useHref(`/admin/datasets/${row.id}`);

    return (
        <MenuList>
            <MenuItemLink
                icon={<ViewIcon />}
                href={datasetHref}
            >
                View details
            </MenuItemLink>
        </MenuList>
    );
};

export const AdminPage_Datasets: React.FC = () => {
    const [limit, setLimit] = useState<number>(10);
    const [offset, setOffset] = useState<number>(0);
    const { data, isLoading, isFetching, error, refetch } = useAdminDatasetsListDatasets({
        queryParams: { offset, limit },
    });

    const navigate = useNavigate();

    if (error) {
        return <MaybeErrorAlert error={error} />;
    }
    if (isLoading) {
        return <Spinner tall />;
    }

    return (
        <>
            <Stack rowGap="XL">
                <Stack horizontal columnGap="L">
                    <PageTitle>Datasets</PageTitle>
                    <Button
                        appearance="subtle"
                        icon={<ArrowClockwiseRegular />}
                        onClick={() => refetch().then()}
                    />
                </Stack>
                <Text size={300} style={{ color: 'var(--colorNeutralForeground2)' }}>
                    Total: {data.total} datasets
                </Text>
                <Stack rowGap="S">
                    <ProgressBar hide={!isFetching} />
                    <SortableTable
                        columns={COLUMNS}
                        rows={data.items}
                        missingPlaceholder={<span style={{ opacity: 0.5 }}>&#823;</span>}
                        onRenderMenuChildren={(row) => <DatasetMenu row={row} />}
                        showMenuOnRowContextMenu
                        onRowClick={(row) => navigate(`/admin/datasets/${row.id}`)}
                    />
                </Stack>
                <Pagination
                    current={1 + Math.floor(offset / limit)}
                    total={data.total}
                    pageSize={limit}
                    responsive
                    hideOnSinglePage
                    showSizeChanger
                    onChange={(page, pageSize) => {
                        setOffset((page - 1) * pageSize);
                        setLimit(pageSize);
                        document?.body?.scrollTo?.({ top: 0, behavior: "smooth" });
                    }}
                />
            </Stack>
        </>
    );
};
