import * as React from 'react';
import {useState} from 'react';
import {useNavigate, useParams} from 'react-router-dom';
import {
    fetchAdminDatasetsDatasetIdDeleteDataset,
    fetchAdminDatasetsDatasetIdOwnerTransferTransferDataset,
    useAdminDatasetsDatasetIdGetDataset,
} from 'meteoio-platform-client';
import {MaybeErrorAlert} from 'meteoio-ui/src/components/MaybeErrorAlert';
import {Spinner} from 'meteoio-ui/src/components/Spinner';
import {Stack} from 'meteoio-ui/src/layouts/Stack';
import {PageTitle} from 'meteoio-ui/src/components/PageTitle';
import {RepositoryAccessPill} from 'meteoio-ui/src/components/RepositoryAccessPill';
import {Button} from 'meteoio-ui/src/components/Button';
import {DeleteRegular, PersonArrowRightRegular} from '@fluentui/react-icons';
import {useDialogs} from 'meteoio-ui/src/hooks/useDialogs';
import {useQueryClient} from '../_common/backend';
import {FreeTextField} from 'meteoio-ui/src/components/FreeTextField';
import {AsyncActionButton} from 'meteoio-ui/src/components/AsyncActionButton';
import {mergeClasses, Text} from "@fluentui/react-components";
import {useJobViewStyles} from "../jobs/job/styles";
import {RelativeTimeText} from "meteoio-ui/src/components/RelativeTimeText";

export const AdminPage_DatasetDetails: React.FC = () => {
    const styles = useJobViewStyles()

    const { datasetId } = useParams<{ datasetId: string }>();
    const { data: dataset, isLoading, error } = useAdminDatasetsDatasetIdGetDataset({
        pathParams: { datasetId: datasetId! },
    });

    const [newOwnerIdStr, setNewOwnerIdStr] = useState<string | undefined>(undefined);
    const dialogs = useDialogs();
    const queryClient = useQueryClient();
    const navigate = useNavigate()

    const selectedRow = dataset
    const dsId = datasetId

    if (error) {
        return <MaybeErrorAlert error={error} />;
    }
    if (isLoading || !dataset) {
        return <Spinner tall />;
    }

    return <>
        <br />
        <br />
        <Stack rowGap="XL" center maxWidth={500}>
            <Stack horizontal columnGap="XXXL">
                <PageTitle>Dataset details</PageTitle>
            </Stack>
            <Text size={200} className={mergeClasses(styles.boxBorder, styles.statusInfo)}>
                Owner ID: <code>{selectedRow?.owner_user_id}</code><br/>
                Dataset ID: <code>{selectedRow?.id}</code><br/>
                Access: <RepositoryAccessPill access={selectedRow?.access}/><br/>
                {selectedRow?.created_at && <>Created: <RelativeTimeText date={selectedRow.created_at} /><br/></>}
                Title: {selectedRow?.heading?.title}<br/>
                Description: {selectedRow?.heading?.description}<br/>
            </Text>

            <div className={mergeClasses(styles.boxBorder, styles.statusInfo)}>
                {/* newOwnerIdStr === undefined   -> transfer intent not initiated */}
                {/* newOwnerIdStr === ''          -> transfer intent just initiated */}
                {/* !!newOwnerIdStr               -> has some user id for transfer */}
                <Stack horizontal alignItems="end">
                    {newOwnerIdStr === undefined && <Button
                        size="small"
                        danger
                        label="Transfer ownership"
                        icon={<PersonArrowRightRegular/>}
                        appearance="subtle"
                        // secondaryLine="Remove entry and all associated data"
                        onClick={async () => {
                            setNewOwnerIdStr('')
                        }}
                    />}
                    {newOwnerIdStr !== undefined && <Stack grow>
                        <FreeTextField
                            label="New owner user ID"
                            fieldProps={{size: 'small'}}
                            value={newOwnerIdStr}
                            onChange={setNewOwnerIdStr}
                            smallCode
                        />
                    </Stack>}
                    {newOwnerIdStr && <AsyncActionButton
                        size="medium"
                        danger
                        label="Transfer"
                        icon={<PersonArrowRightRegular/>}
                        appearance="secondary"
                        disabled={newOwnerIdStr?.trim?.() === selectedRow.owner_user_id}
                        // secondaryLine="Remove entry and all associated data"
                        onClick={async () => {
                            if (!await dialogs.confirm('Are you sure you want to transfer the dataset to the specified user?', {
                                okType: 'danger',
                                okText: 'Transfer'
                            })) {
                                return
                            }
                            await fetchAdminDatasetsDatasetIdOwnerTransferTransferDataset({
                                pathParams: {datasetId: dsId},
                                queryParams: {
                                    old_owner_user_id: selectedRow.owner_user_id,
                                    new_owner_user_id: newOwnerIdStr.trim(),
                                }
                            })
                            dialogs.message.success('Dataset transferred successfully.')
                            await queryClient.invalidateQueries()
                            setNewOwnerIdStr(undefined)
                        }}
                    />}
                </Stack>
                {newOwnerIdStr?.trim?.() === selectedRow.owner_user_id && <Text block size={200}>
                    The specified user ID matches the current owner user ID.
                </Text>}
                {newOwnerIdStr !== undefined && <Text block size={200}>
                    Please find the user ID of the new owner in the users administration page.
                </Text>}
            </div>

            <div className={mergeClasses(styles.boxBorder, styles.statusInfo)}>
                <AsyncActionButton
                    size="small"
                    danger
                    icon={<DeleteRegular/>}
                    label="Delete Dataset"
                    appearance="subtle"
                    // secondaryLine="Remove entry and all associated data"
                    onClick={async () => {
                        dialogs.message.warning('You are going to permanently delete the dataset.', 5)
                        await new Promise(resolve => setTimeout(resolve, 2000))
                        if (!await dialogs.confirm('Are you sure you want to permanently delete this dataset? This action cannot be undone.', {
                            okType: 'danger',
                            okText: 'Delete'
                        })) {
                            return
                        }
                        await fetchAdminDatasetsDatasetIdDeleteDataset({
                            pathParams: {datasetId: dsId},
                            queryParams: {owner_user_id_for_check: selectedRow.owner_user_id}
                        })
                        dialogs.message.success('Dataset deleted successfully.')
                        await queryClient.invalidateQueries()
                        setNewOwnerIdStr(undefined)
                        navigate(`/admin/datasets`)
                    }}
                />
            </div>
        </Stack>
        <br />
        <br />
    </>
};