import {
    useInternalDatasetsDatasetIdConfigTxsTxIdJobEnvGetJobEnv,
    useInternalSystemEnvReadSystemEnv,
    usePublicMeJobEnvConfigLoadUserProfileJobEnvConfig
} from "meteoio-platform-client";
import {KeyValueEntry} from "meteoio-ui/src/components/KeyValueTable";
import { useMemo } from "react";

export type Level = 'system' | 'user' | 'dataset' | 'cron'

export interface CompiledKeyValueEntry extends KeyValueEntry {
    source: Level
    overriddenById?: string
}
export function useCompiledEnvEntries(level: Level, local: KeyValueEntry[] | undefined = undefined, datasetId: string | undefined = undefined, datasetTxId: string | undefined = undefined): {
    entries: CompiledKeyValueEntry[],
    error?: Error | null,
    isLoading: boolean,
} {
    // NOTE: The levels are: system, user, dataset, cron.
    //       There's nothing below cron, where the local settings are passed as arg, so, there's no cronId argument.
    //       There's nothing above system level, where this hook is not necessary, but it's still possible to use it.

    // NOTE: Will fetch non-local level data. For data of this `level`, must use `local`!
    const system = useInternalSystemEnvReadSystemEnv({}, {
        enabled: level != 'system',
    })
    const user = usePublicMeJobEnvConfigLoadUserProfileJobEnvConfig({}, {
        enabled: level != 'user',
    })
    const dataset = useInternalDatasetsDatasetIdConfigTxsTxIdJobEnvGetJobEnv({
        pathParams: {datasetId, txId: datasetTxId},
    }, {
        enabled: !!datasetId && !!datasetTxId && level != 'dataset',
    })

    const upperLevelsLevels = useMemo<Level[]>(() => {
        if (level == 'system') {
            return []
        }
        if (level == 'user') {
            return ['system']
        }
        if (level == 'dataset') {
            return ['system', 'user']
        }
        if (level == 'cron') {
            return ['system', 'user', 'dataset']
        }
        return []
    }, [level])

    // NOTE: For data of this `level`, must use `local`, to avoid duplicates/overrides!

    const entries = useMemo(() => {
        const map: Map<string, CompiledKeyValueEntry> = new Map()
        // NOTE: The map will hold the last encountered entry, which may then be overridden.
        const entries: CompiledKeyValueEntry[] = []

        const handle: (source: Level) => ((e: KeyValueEntry) => void) = source => e => {
            const old = map.get(e.key)
            if (old) {
                old.overriddenById = e.id
            }
            const newEntry: CompiledKeyValueEntry = {...e, source}
            map.set(e.key, newEntry)
            entries.push(newEntry)
        }

        if (upperLevelsLevels.includes('system')) {
            system.data?.entries?.forEach?.(handle('system'))
        }
        if (upperLevelsLevels.includes('user')) {
            user.data?.entries?.forEach?.(handle('user'))
        }
        if (upperLevelsLevels.includes('dataset')) {
            dataset.data?.entries?.forEach?.(handle('dataset'))
        }
        local?.forEach?.(handle(level))

        return entries
    }, [system.data, user.data, dataset.data, local, upperLevelsLevels])

    return {
        isLoading: system.isLoading || user.isLoading || dataset.isLoading,
        error: (system.error ?? user.error ?? dataset.error) as undefined as (Error | undefined),
        entries,
    }
}
