import React, {useState} from 'react';
import {Button, Input, Select, Space} from 'antd';

interface Filters {
    [key: string]: string;
}

interface CustomFilterProps {
    setFilter: React.Dispatch<React.SetStateAction<Filters>>;
    withUserIdFilterOption?: boolean
}

const EVENT_TYPE_OPTIONS = [  // See also python class EventType
    { value: 'dataset_cron_job', label: 'Dataset Cron Job' },
    { value: 'dataset_manual_job', label: 'Dataset Manual Job' },
    { value: 'data_owner_privilege_request', label: 'Data Owner Privilege Request' },
];

export const EventFilterFreeInputSetter: React.FC<CustomFilterProps> = ({ setFilter, withUserIdFilterOption }) => {
    const [selectedFilterKey, setSelectedFilterKey] = useState<string | undefined>(withUserIdFilterOption ? undefined : 'fingerprint');
    const [filterValue, setFilterValue] = useState<string>("");

    const handleAddCustomFilter = () => {
        if (selectedFilterKey && filterValue) {
            setFilter((prev) => ({ ...prev, [selectedFilterKey]: filterValue }));
            setSelectedFilterKey(undefined);
            setFilterValue("");
        }
    };

    const isEventTypeFilter = selectedFilterKey === 'event_type';

    return <>
            <Space>
                <Select
                    placeholder="Select filter type"
                    value={selectedFilterKey}
                    onChange={(value) => setSelectedFilterKey(value)}
                    // style={{ width: 200 }}
                    popupMatchSelectWidth={false}
                    options={[
                        ...(withUserIdFilterOption ? [{ value: 'from_user_id', label: 'User ID' }] : []),
                        { value: 'fingerprint', label: 'Fingerprint' },
                        { value: 'event_type', label: 'Event Type' },
                    ]}
                    size="small"
                />
                {isEventTypeFilter ? (
                    <Select
                        placeholder="Select event type"
                        title="Select event type"
                        value={filterValue}
                        onChange={(value) => setFilterValue(value)}
                        style={{ width: 200 }}
                        popupMatchSelectWidth={false}
                        options={EVENT_TYPE_OPTIONS}
                        size="small"
                    />
                ) : (
                    <Input
                        placeholder="Enter filter value"
                        value={filterValue}
                        onChange={(e) => setFilterValue(e.target.value)}
                        onPressEnter={handleAddCustomFilter}
                        style={{ width: 200 }}
                        size="small"
                    />
                )}
                <Button
                    type="primary"
                    onClick={handleAddCustomFilter}
                    size="small"
                >
                    Filter
                </Button>
            </Space>
    </>
};
