import * as React from 'react'
import {useEffect, useState} from 'react'
import {KeyValueEntry, KeyValueTable} from "meteoio-ui/src/components/KeyValueTable";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {SaveRegular} from "@fluentui/react-icons";
import {useDialogs} from "meteoio-ui/src/hooks/useDialogs";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {Alert} from "meteoio-ui/src/components/Alert";
import {getMaybeErrorMessageString, MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {ACDDMetadataForm} from "./ACDDMetadataForm";
import {ACDD_FIELD_DEFINITIONS, EnvLevelKey, FieldDefinition} from "./ACDDMetadataFields";
import {Segmented} from "antd";
import {CompiledKeyValueEntry, useCompiledEnvEntries} from './useCompiledEnvEntries';
import {Column, SortableTable} from "meteoio-ui/src/components/SortableTable";
import {Link, useParams} from "react-router-dom";
import {address_book} from "../address_book";
import {Switch} from "antd";
import {Text, tokens} from "@fluentui/react-components";

const ENV_LEVEL_TO_ACDD_FIELDS: Record<EnvLevelKey, FieldDefinition[]> = {
    system: ACDD_FIELD_DEFINITIONS.filter(field => !!field.levels?.includes('system')),
    user: ACDD_FIELD_DEFINITIONS.filter(field => !!field.levels?.includes('user')),
    dataset: ACDD_FIELD_DEFINITIONS.filter(field => !!field.levels?.includes('dataset')),
    cron: ACDD_FIELD_DEFINITIONS.filter(field => !!field.levels?.includes('cron')),
}

const COMPILED_COLUMNS: Column<CompiledKeyValueEntry>[] = [
    {
        field: 'source',
        title: 'Source',
        onRenderCellContent: row => {
            if (row.source === 'system') {
                return <Link to={address_book.admin.job_env}>
                    System
                </Link>
            }
            if (row.source === 'user') {
                return <Link to={address_book.user.job_env}>
                    User profile
                </Link>
            }
            if (row.source === 'dataset') {
                return <DsLink/>
            }
            if (row.source === 'cron') {
                return <>This cron job</>
            }
            return null
        }
    },
    {
        field: 'key',
        title: 'Variable name',
        onRenderCellContent: row => <code
            style={{
                textDecoration: row.overriddenById ? 'line-through' : undefined,
                opacity: row.overriddenById ? 0.5 : undefined,
            }}
        >
            {row.key}
        </code>
    },
    {
        field: 'value',
        title: 'Value',
        onRenderCellContent: row => <code
            style={{
                whiteSpace: 'pre-wrap',
                textDecoration: row.overriddenById ? 'line-through' : undefined,
                // textDecorationColor: row.overriddenById ? '#7778' : undefined,
                opacity: row.overriddenById ? 0.5 : undefined,
            }}
        >
            {row.value}
        </code>
    }
]

export const EnvSettingsEditor: React.FC<{
    headerExtra?: React.ReactNode
    disabled?: boolean
    onLoad?: () => Promise<KeyValueEntry[]>
    onStore?: (entries: KeyValueEntry[]) => Promise<void>
    onDiscard?: () => void
    extraDependencyKey?: string
    immediateLoadedValue?: KeyValueEntry[]  // This is to allow reactive rendering without onLoad
    onChangeImmediate?: (entries: KeyValueEntry[]) => void
    useImmediateOnly?: boolean // to be used with onChangeImmediate probably
    level?: EnvLevelKey
    compilingDatasetId?: string
    compilingDatasetTxId?: string
}> = props => {
    const [loadError, setLoadError] = useState<unknown | Error>(undefined)
    const [tab, setTab] = useState('env')
    const [draftEntries, setDraftEntries] = useState<KeyValueEntry[]>(undefined)
    const [loadedEntries, setLoadedEntries] = useState<KeyValueEntry[]>(undefined)
    const savedEntries = loadedEntries ?? props.immediateLoadedValue
    const shownEntries = props.useImmediateOnly ? props.immediateLoadedValue : (draftEntries ?? savedEntries)
    const hasChanges = shownEntries !== savedEntries

    const acdd_fields = React.useMemo(() => ENV_LEVEL_TO_ACDD_FIELDS[props.level ?? 'system'], [props.level])

    const {confirm, message} = useDialogs()

    const compiled = useCompiledEnvEntries(props.level, shownEntries, props.compilingDatasetId, props.compilingDatasetTxId)
    const [isCompiledOverridesVisible, setIsCompiledOverridesVisible] = useState<boolean>(false)

    useEffect(() => {
        setLoadedEntries(undefined)
        setDraftEntries(undefined)
    }, [props.extraDependencyKey]);

    useEffect(() => {
        let cancelled = false;
        if (loadedEntries === undefined) {
            setLoadError(undefined)
            props.onLoad?.()?.then?.((entries) => {
                if (!cancelled) {
                    setLoadedEntries(entries)
                }
            })?.catch?.(err => {
                if (!cancelled) {
                    setLoadError(err)
                }
            })
        }
        return () => {
            cancelled = true;
        }
    }, [loadedEntries, props.extraDependencyKey]);

    useEffect(() => {
        // Support external state instead of onStore
        if (props.onChangeImmediate) {
            props.onChangeImmediate(draftEntries)
        }
    }, [draftEntries]);

    if (loadError) {
        return <MaybeErrorAlert error={loadError}/>
    }

    if (shownEntries === undefined) {
        return <Spinner tall label="Loading environment variables..."/>
    }

    return <>
        <Stack rowGap="L">
            <Stack justifyContent="space-between" horizontal>
                <PageSectionTitle>Environment variables {shownEntries ? <>({shownEntries?.length ?? 0})</> : null}</PageSectionTitle>
                {props.headerExtra}
            </Stack>
            <Stack rowGap="XL">
                <Segmented  // alternative: TabListWithOverflow
                    // block
                    size="middle"
                    value={tab}
                    options={[
                        {
                            value: 'env',
                            label: <>All entries</>,
                        },
                        {
                            value: 'acdd',
                            label: <>ACDD mask</>,
                        },
                        {
                            value: 'other',
                            label: <>Custom entries</>,
                        },
                        {
                            value: 'result',
                            label: <>Result</>,
                        },
                    ]}
                    onChange={value => setTab(value as typeof tab)}
                />
                {tab === 'env' && <KeyValueTable
                    keyLabel="Variable name"
                    entries={shownEntries ?? []}
                    onChange={setDraftEntries}
                    disabled={props.disabled}
                />}
                {tab === 'acdd' && <ACDDMetadataForm
                    fields={acdd_fields}
                    entries={shownEntries ?? []}
                    setEntries={setDraftEntries}
                    // placeholder={props.level == 'system' ? '(not set)' : '(not set, may be inherited)'}
                />}
                {tab === 'other' && <KeyValueTable
                    keyLabel="Variable name"
                    entries={(shownEntries ?? [])?.filter(e => !acdd_fields.find(f => f.key == e.key))}
                    onChange={setDraftEntries}
                    disabled={props.disabled}
                />}
                {tab === 'result' && <>
                    <MaybeErrorAlert error={compiled.error}/>
                    <Stack rowGap="None">
                        <div style={{marginTop: -12, height: 0, overflow: 'hidden'}}>&nbsp;</div>
                        <div style={{position: 'relative', zIndex: 1}}>
                            <Stack horizontal justifyContent="end" alignItems="center" columnGap="XS">
                                <label>
                                    <Text size={100} style={{color: tokens.colorNeutralForeground3}}>Show overrides &nbsp;</Text>
                                    <Switch
                                        onChange={checked => setIsCompiledOverridesVisible(checked)}
                                        checked={isCompiledOverridesVisible}
                                        checkedChildren="On"
                                        unCheckedChildren="Off"
                                        size="small"
                                    />
                                </label>
                            </Stack>
                        </div>
                        <SortableTable
                            style={{marginTop: -10}}
                            size="extra-small"
                            rows={isCompiledOverridesVisible ? compiled.entries : compiled.entries.filter(e => !e.overriddenById)}
                            columns={COMPILED_COLUMNS}
                        />
                    </Stack>
                </>}
            </Stack>

            {/* TODO?: Move these buttons / banner above ? Maybe sticky? */}
            {props.useImmediateOnly ? <></> : <>

                <div>&nbsp;</div>

                {hasChanges ? <Stack horizontal justifyContent="flex-end">
                    <AsyncActionButton
                        // icon={<SaveRegular/>}
                        // appearance="primary"
                        label={"Discard changes"}
                        onClick={async () => {
                            if (hasChanges && !await confirm('Are you sure you want to discard your pending changes?')) {
                                return
                            }
                            props.onDiscard?.()
                            setLoadedEntries(undefined)
                            setDraftEntries(undefined)
                        }}
                    />
                    {props.onStore && <AsyncActionButton
                        icon={<SaveRegular/>}
                        appearance="primary"
                        label="Save"
                        disabled={!hasChanges || props.disabled}
                        onClick={async () => {
                            try {
                                await props.onStore(draftEntries)
                                props.onDiscard?.()
                                setLoadedEntries(undefined)
                                setDraftEntries(undefined)
                            }
                            catch (err) {
                                message.error({
                                    content: getMaybeErrorMessageString(err)
                                })
                            }
                        }}
                    />}
                </Stack> : <Stack horizontal justifyContent="flex-end">
                    <Alert level="low_info" size="small">There are no pending changes</Alert>
                    {props.onDiscard && <AsyncActionButton
                        // icon={<CloseRegular/>}
                        // appearance="primary"
                        label={"Close"}
                        onClick={async () => {
                            props.onDiscard()
                            setLoadedEntries(undefined)
                            setDraftEntries(undefined)
                        }}
                    />}
                </Stack>}
            </>}
        </Stack>
    </>
}


const DsLink: React.FC = () => {
    const {datasetId} = useParams()
    return <Link to={address_book.datasets.settings.job_env(datasetId)}>
        Dataset
    </Link>
}
