import React, {Dispatch, SetStateAction, useCallback} from 'react';
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {FieldDefinition} from "./ACDDMetadataFields";
import {ACDDField} from "./ACDDField";
import {ExtLink} from "meteoio-ui/src/components/ExtLink";

export interface KeyValueEntry {
    id: string;
    key: string;
    value: string;
}

export interface ACDDMetadataFormProps {
    fields: FieldDefinition[]
    entries?: KeyValueEntry[]
    setEntries: Dispatch<SetStateAction<KeyValueEntry[] | undefined>>
    placeholder?: string
}

/**
 * Parent form maps definitions → memoized ACDDField rows.
 * We also memoize handleChange so its reference stays stable.
 */
export const ACDDMetadataForm: React.FC<ACDDMetadataFormProps> = (props) => {
    const entries = props.entries === undefined ? [] : props.entries;

    // Stable handler for all fields
    const handleChange = useCallback(
        (key: string, newVal: string) => {
            props.setEntries(current => {

                // NOTE!!!: prop.entries is initially different from the draft value received as current from props.setEntries
                const list = current ?? props.entries;  // This is important and tricky!

                // NOTE: See note above!

                // remove if empty
                if (!newVal) {
                    return list.filter(e => e.key !== key);
                }
                // update or append
                let found = false;
                const updated = list.map(e => {
                    if (e.key === key) {
                        found = true;
                        return {...e, value: newVal};
                    }
                    return e;
                });
                return found
                    ? updated
                    : [
                        ...updated,
                        {id: crypto.randomUUID(), key, value: newVal},
                    ];
            });
        },
        [props.setEntries]
    );

    return (
        <Stack rowGap="M">
            {props.fields.map(field => (
                <ACDDField
                    key={field.key}
                    field={field}
                    value={entries.find(e => e.key === field.key)?.value}
                    onChange={handleChange}
                    placeholder={props.placeholder}
                />
            ))}

            <small style={{opacity: 0.7}}>
                For more information, see the <ExtLink href="https://wiki.esipfed.org/Attribute_Convention_for_Data_Discovery_1-3">Attribute Convention for Data Discovery</ExtLink>
            </small>
        </Stack>
    );
};
