export type EnvLevelKey = 'system' | 'user' | 'dataset' | 'cron'

export interface FieldDefinition {
    key: string // env key according to MeteoIO
    attributeACDD?: string // attribute code name according to ACDD specs
    label: string
    description: string
    type: 'text' | 'textarea' | 'number' | 'date' | 'tags' | 'enum'
    numericStep?: number
    enumOptions?: { value: string, text: string, label: string }[]  // for enum only
    required?: boolean  // true iff the field is required according to met.no submission guide
    systematic?: boolean  // the field should be systematically set by the platform

    levels: EnvLevelKey[]  // NOTE: Very important! This defines at which level the field will be visible in the UI.
}

/**
 * ACDD metadata fields as per MET-NO CF/ACDD submission guide.
 * https://adc.met.no/submit-data-as-netcdf-cf
 *  https://wiki.esipfed.org/Attribute_Convention_for_Data_Discovery_1-3
 */
export const ACDD_FIELD_DEFINITIONS: FieldDefinition[] = [
    {
        key: 'ACDD_ID',
        attributeACDD: 'id',
        label: 'ID',
        description:
            'Unique dataset identifier, e.g. DOI or other persistent identifier.',
        type: 'text',
        required: true,
        systematic: true,
        levels: [],
    },

    // ACDD-1.3: 'naming_authority' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_NAMING_AUTHORITY',
        attributeACDD: 'naming_authority',
        label: 'Naming Authority',
        description:
            'Organization providing the dataset identifiers (e.g. reverse-DNS).',
        type: 'text',
        required: true,
        levels: ['system'],
    },

    // ACDD-1.3: 'title' is *Highly Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_TITLE',
        attributeACDD: 'title',
        label: 'Title',
        description: 'Concise, human-readable dataset title.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset'],
    },

    // ACDD-1.3: 'summary' is *Highly Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_SUMMARY',
        attributeACDD: 'summary',
        label: 'Summary',
        description: 'Paragraph-length abstract describing the dataset.',
        type: 'textarea',
        required: true,
        levels: ['cron', 'dataset'],
    },

    // ACDD-1.3: 'keywords' is *Highly Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_KEYWORDS',
        attributeACDD: 'keywords',
        label: 'Keywords',
        description: 'Comma-separated list of terms from GCMD Science Keywords.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset'],
    },

    // ACDD-1.3: 'keywords_vocabulary' is *Suggested*
    // MET submission guide: required (extension)
    {
        key: 'ACDD_KEYWORDS_VOCABULARY',
        attributeACDD: 'keywords_vocabulary',
        label: 'Keywords Vocabulary',
        description: 'Identifier of the controlled vocabulary used for keywords.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset'],
    },

    /*
     *  NOTE: Omitted Geospatial and Temporal Coverage Attributes
     *        I commented out the global attributes geospatial_lat_min, geospatial_lat_max, geospatial_lon_min, and geospatial_lon_max.
     *        ACDD-1.3 explicitly recommends these for defining a spatial extent,
     *        and MET-NO’s CF/ACDD submission guide requires them for proper spatial metadata.
     *        I also removed time_coverage_start and time_coverage_end.
     *        ACDD-1.3 lists these as Recommended to describe the dataset’s time span,
     *        and MET-NO mandates them to capture temporal coverage.
     *        These fields shall not be assigned by the user but systematically computed.
     */
    //     {
    //         attributeACDD: 'geospatial_lat_min',
    //         label: 'Southernmost Latitude',
    //         description: 'Southernmost latitude bound (decimal degrees north).',
    //         type: 'number',
    //         numericStep: 0.01,
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },
    //     {
    //         attributeACDD: 'geospatial_lat_max',
    //         label: 'Northernmost Latitude',
    //         description: 'Northernmost latitude bound (decimal degrees north).',
    //         type: 'number',
    //         numericStep: 0.01,
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },
    //     {
    //         attributeACDD: 'geospatial_lon_min',
    //         label: 'Westernmost Longitude',
    //         description:
    //             'Westernmost longitude bound (decimal degrees east; negative westwards).',
    //         type: 'number',
    //         numericStep: 0.01,
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },
    //     {
    //         attributeACDD: 'geospatial_lon_max',
    //         label: 'Easternmost Longitude',
    //         description:
    //             'Easternmost longitude bound (decimal degrees east; negative westwards).',
    //         type: 'number',
    //         numericStep: 0.01,
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },
    //     {
    //         attributeACDD: 'time_coverage_start',
    //         label: 'Time Coverage Start',
    //         description: 'ISO 8601 timestamp (UTC) of the first data point.',
    //         type: 'date',
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },
    //     {
    //         attributeACDD: 'time_coverage_end',
    //         label: 'Time Coverage End',
    //         description:
    //             'ISO 8601 timestamp (UTC) of the last data point (omit for ongoing datasets).',
    //         type: 'date',
    //         required: true,
    //         systematic: true,
    //         levels: [],
    //     },

    /*
     *
     *  NOTE: The following fields shall not be assigned by the user but systematically computed.
     */
    //     {
    //         key: 'ACDD_CONVENTIONS',  // NOTE: This does not seem to be part of MeteoIO accepted env keys! But it's required!
    //         attributeACDD: 'Conventions',
    //         label: 'Conventions',
    //         description:
    //             'Comma-separated list of conventions followed (must include “ACDD-1.3”).',
    //         type: 'text',  // tags
    //         required: true,
    //         levels: ['cron', 'dataset', 'user'],
    //     },
    //     {
    //         attributeACDD: 'history',
    //         label: 'History',
    //         description: 'Audit trail of dataset modifications.',
    //         type: 'textarea',
    //         required: true,
    //         levels: ['cron', 'dataset'],
    //     },
    //     {
    //         attributeACDD: 'date_created',
    //         label: 'Date Created',
    //         description: 'ISO 8601 date/time when this version was created.',
    //         type: 'date',
    //         required: true,
    //         levels: ['cron', 'dataset'],
    //     },

    // ACDD-1.3: 'creator_type' is *Suggested*
    // MET submission guide: required
    {
        key: 'ACDD_CREATOR_TYPE',
        attributeACDD: 'creator_type',
        label: 'Creator Type',
        description: 'Type of creator (person, group, or institution).',
        type: 'enum',
        enumOptions: [
            {value: 'person', text: 'person', label: 'person'},
            {value: 'group', text: 'group', label: 'group'},
            {value: 'institution', text: 'institution', label: 'institution'},
            {value: 'position', text: 'position', label: 'position'},
        ],
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'creator_institution' is *Suggested*
    // MET submission guide: required
    {
        key: 'ACDD_CREATOR_INSTITUTION',
        attributeACDD: 'creator_institution',
        label: 'Creator Institution',
        description: 'Institution of the dataset creator.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'creator_name' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_CREATOR',
        attributeACDD: 'creator_name',
        label: 'Creator Name',
        description: 'Name of the dataset creator.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'creator_email' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_CREATOR_EMAIL',
        attributeACDD: 'creator_email',
        label: 'Creator Email',
        description: 'Email address of the dataset creator.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'creator_url' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_CREATOR_URL',
        attributeACDD: 'creator_url',
        label: 'Creator URL',
        description: 'URL of the dataset creator.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'publisher_name' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_PUBLISHER',
        attributeACDD: 'publisher_name',
        label: 'Publisher Name',
        description: 'Name of the publisher.',
        type: 'text',
        required: true,
        levels: ['system'],
    },

    // ACDD-1.3: 'publisher_email' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_PUBLISHER_EMAIL',
        attributeACDD: 'publisher_email',
        label: 'Publisher Email',
        description: 'Email address of the publisher.',
        type: 'text',
        required: true,
        levels: ['system'],
    },

    // ACDD-1.3: 'publisher_url' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_PUBLISHER_URL',
        attributeACDD: 'publisher_url',
        label: 'Publisher URL',
        description: 'URL of the publisher.',
        type: 'text',
        required: true,
        levels: ['system'],
    },

    // ACDD-1.3: 'project' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_PROJECT',
        attributeACDD: 'project',
        label: 'Project',
        description: 'Name(s) of the project(s) responsible for the data.',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // ACDD-1.3: 'license' is *Recommended*
    // MET submission guide: required
    {
        key: 'ACDD_LICENSE',
        attributeACDD: 'license',
        label: 'License',
        description:
            'URL or SPDX identifier of the data license (e.g. “CC-BY-4.0”).',
        type: 'text',
        required: true,
        levels: ['cron', 'dataset', 'user', 'system'],
    },

    // **Extension**: 'summary_file' is *not* in ACDD-1.3
    // MET submission guide: required for replacing 'summary'
    {
        key: 'ACDD_SUMMARY_FILE',
        // attributeACDD: 'summary_file', // not part of ACDD specs
        label: 'Summary File',
        description:
            'A file containing a description of the dataset; it overwrites the value of Summary if present.',
        type: 'text',
        required: true,
        levels: ['dataset'],
    },

    // ACDD-1.3: 'comment' is *Recommended*
    // MET submission guide: optional
    {
        key: 'ACDD_COMMENT',
        attributeACDD: 'comment',
        label: 'Comment',
        description: 'Miscellaneous information about the dataset.',
        type: 'textarea',
        // required: true,
        levels: ['dataset', 'user', 'cron'],
    },

    // ACDD-1.3: 'acknowledgement' is *Recommended*
    // MET submission guide: optional
    {
        key: 'ACDD_ACKNOWLEDGEMENT',
        attributeACDD: 'acknowledgement',
        label: 'Acknowledgement',
        description:
            'Acknowledgment for the various types of support for the project that produced this data.',
        type: 'textarea',
        // required: true,
        levels: ['dataset', 'user'],
    },

    // **Extension**: 'wigos_id' is not part of ACDD-1.3
    // MET submission guide: optional
    {
        key: 'WIGOS_ID',
        // attributeACDD: 'wigos_id',
        label: 'WIGOS ID',
        description:
            'Although this is not an ACDD key, it can be very useful in linking datasets together through their WIGOS ID.',
        type: 'text',
        // required: true,
        levels: ['dataset', 'cron'],
    },

    // ACDD-1.3: 'references' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_REFERENCES',
        attributeACDD: 'references',
        label: 'References',
        description:
            'Published or web-based references that describe the data or methods used to produce it.',
        type: 'text',
        // required: true,
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'metadata_link' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_METADATA_LINK',
        attributeACDD: 'metadata_link',
        label: 'Metadata Link',
        description: 'A URL/DOI that gives more complete metadata.',
        type: 'text',
        // required: true, // not required; suggested by acdd specs
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'source' is *Recommended*
    // MET submission guide: optional
    {
        key: 'ACDD_SOURCE',
        attributeACDD: 'source',
        label: 'Source',
        description: 'The source and method of production of the original data.',
        type: 'text',
        // required: true,
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'program' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_PROGRAM',
        attributeACDD: 'program',
        label: 'Program',
        description: 'The overarching program(s) of which the dataset is a part.',
        type: 'text',
        // required: true, // not required; suggested by acdd specs
        levels: ['dataset', 'user'],
    },

    // **Extension**: 'activity_type' not in ACDD-1.3
    // MET submission guide: optional
    {
        key: 'ACDD_ACTIVITY_TYPE',
        // attributeACDD: 'activity_type',  // Not part of ACDD, MET extension.
        label: 'Activity Type',
        description:
            'Controlled vocabulary used to describe activity types (see spec).',
        type: 'enum',
        enumOptions: [
            {value: 'aircraft', text: 'aircraft', label: 'aircraft'},
            {value: 'space borne instrument', text: 'space borne instrument', label: 'space borne instrument'},
            {value: 'numerical simulation', text: 'numerical simulation', label: 'numerical simulation'},
            {value: 'climate indicator', text: 'climate indicator', label: 'climate indicator'},
            {
                value: 'in situ land-based station',
                text: 'in situ land-based station',
                label: 'in situ land-based station'
            },
            {
                value: 'in situ ship-based station',
                text: 'in situ ship-based station',
                label: 'in situ ship-based station'
            },
            {
                value: 'in situ ocean fixed station',
                text: 'in situ ocean fixed station',
                label: 'in situ ocean fixed station'
            },
            {
                value: 'in situ ocean moving station',
                text: 'in situ ocean moving station',
                label: 'in situ ocean moving station'
            },
            {value: 'in situ ice-based station', text: 'in situ ice-based station', label: 'in situ ice-based station'},
            {value: 'interview/questionnaire', text: 'interview/questionnaire', label: 'interview/questionnaire'},
            {value: 'maps/charts/photographs', text: 'maps/charts/photographs', label: 'maps/charts/photographs'},
        ],
        // required: true,
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'publisher_type' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_PUBLISHER_TYPE',
        attributeACDD: 'publisher_type',
        label: 'Publisher Type',
        description:
            'Type of publisher(s), chosen from (person|group|institution|position).',
        type: 'enum',
        enumOptions: [
            {value: 'person', text: 'person', label: 'person'},
            {value: 'group', text: 'group', label: 'group'},
            {value: 'institution', text: 'institution', label: 'institution'},
            {value: 'position', text: 'position', label: 'position'},
        ],
        // required: true,
        levels: ['system'],
    },

    // ACDD-1.3 'contributor_name' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_CONTRIBUTOR',
        attributeACDD: 'contributor_name',
        label: 'Contributor',
        description:
            'Name of the person(s) or entity(ies) who has/have contributed to the data set.',
        type: 'text',
        // required: true,
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'contributor_role' is *Suggested*
    // MET submission guide: optional
    {
        key: 'ACDD_CONTRIBUTOR_ROLE',
        attributeACDD: 'contributor_role',
        label: 'Contributor Role',
        description: 'Role of the contributor(s).',
        type: 'text',
        // required: true,
        levels: ['dataset', 'user'],
    },

    // ACDD-1.3: 'institution' is *Recommended*
    // MET submission guide: optional
    {
        key: 'ACDD_INSTITUTION',
        attributeACDD: 'institution',
        label: 'Institution',
        description:
            'Institution providing the dataset (legacy field, only one entry).',
        type: 'text',
        // required: true,
        levels: ['system', 'user', 'dataset'],
    },

    // ACDD-1.3: 'product_version' is *Suggested*
    // MET submission guide: systematic versioning
    {
        key: 'ACDD_PRODUCT_VERSION',
        attributeACDD: 'product_version',
        label: 'Product Version',
        description:
            'Version identifier of the data file or product as assigned by the data creator.',
        type: 'text',
        systematic: true,
        // required: true,
        // levels: ['cron'],
        levels: [],
    },

    // ACDD-1.3: 'processing_level' is *Recommended*
    // MET submission guide: optional
    {
        key: 'ACDD_PROCESSING_LEVEL',
        attributeACDD: 'processing_level',
        label: 'Processing Level',
        description: 'A textual description of the processing level.',
        type: 'text',
        // required: true,
        levels: ['cron', 'dataset'],
    },

    // **Extension**: 'operational_status' not in ACDD-1.3
    // MET submission guide: optional
    {
        key: 'ACDD_OPERATIONAL_STATUS',
        // attributeACDD: 'operational_status',  // Not part of ACDD, MET extension.
        label: 'Operational Status',
        description: 'Controlled vocabulary used to describe the operational status.',
        type: 'enum',
        enumOptions: [
            {value: 'operational', text: 'operational', label: 'operational'},
            {value: 'pre-operational', text: 'pre-operational', label: 'pre-operational'},
            {value: 'experimental', text: 'experimental', label: 'experimental'},
            {value: 'scientific', text: 'scientific', label: 'scientific'},
        ],
        // required: true,
        levels: ['system', 'dataset', 'cron'],
    },
]
