import {FieldDefinition} from "./ACDDMetadataFields";
import React, {useCallback, useMemo} from "react";
import {Select} from "antd";
import {DateTimeIsoPicker} from "meteoio-ui/src/components/DateTimeIsoPicker";
import {FieldDescription} from "meteoio-ui/src/components/FieldDescription";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {ControlLabel} from "meteoio-ui/src/components/ControlLabel";
import {Input, SpinButton, Textarea} from "@fluentui/react-components";

/** Props for a single field row */
export interface ACDDFieldProps {
    field: FieldDefinition;
    value: string | undefined;
    onChange: (key: string, newVal: string) => void;
    placeholder?: string
}


/**
 * Memoized field row.
 * It only re-renders when `value` changes (or field config does, which is static).
 */
export const ACDDField: React.FC<ACDDFieldProps> = React.memo(
    (props) => {

        // Stable callback to only change when onChange or key changes
        const handleValueChange = useCallback(
            (newVal: string) => props.onChange(props.field.key, newVal),
            [props.field.key, props.onChange]
        );

        // Build the widget once per [type, value, callback]
        const widget = useMemo(() => {
            const placeholder = props.placeholder ?? '(not set)';
            switch (props.field.type) {
                case 'text':
                    return (
                        <Input
                            placeholder={placeholder}
                            value={props.value as string ?? ''}
                            onChange={e => handleValueChange(e.target.value)}
                        />
                    );
                case 'textarea':
                    return (
                        <Textarea
                            value={props.value ?? ''}
                            rows={4}
                            resize="vertical"
                            placeholder={placeholder}
                            onChange={e => handleValueChange(e.target.value)}
                        />
                    );
                case 'number':
                    return (
                        // <InputNumber
                        //     style={{width: '100%'}}
                        //     value={props.value !== undefined ? Number(props.value) : null}
                        //     placeholder={placeholder}
                        //     step={props.field.numericStep ?? 1}
                        //     onChange={num =>
                        //         handleValueChange(
                        //             num !== undefined ? num.toString() : ''
                        //         )
                        //     }
                        // />
                        <SpinButton // fluentui
                            style={{width: '100%'}}
                            value={props.value !== undefined && props.value !== null && props.value !== '' ? Number(props.value) : null}
                            placeholder={placeholder}
                            step={props.field.numericStep ?? 1}
                            onChange={(_, v) => {
                                // NOTE: If a string is set in the env, then NaN will be displayed in this input.
                                //       Then, if you try to change by typing in, you get v.value NaN and v.displayValue undefined.
                                // NOTE: If a valid numeric value is set, then you get v.value undefined and v.displayValue well-defined.
                                // NOTE: If a string is typed, then you get v.value undefined and v.displayValue the typed string.
                                //       Then, if a number is typed, NaN remains...?
                                const x = parseSimpleValidFloatOrNull(v.value) ?? parseSimpleValidFloatOrNull(v.displayValue)
                                handleValueChange(
                                    x !== undefined && x !== null ? x.toString() : ''
                                )
                            }}
                        />
                    );
                case 'date':
                    return (
                        // <DatePicker
                        //     showTime
                        //     style={{width: '100%'}}
                        //     format="YYYY-MM-DDTHH:mm:ss[Z]"
                        //     value={props.value ? dayjs(props.value) : undefined}
                        //     placeholder={placeholder}
                        //     onChange={(date) =>
                        //         handleValueChange(date?.toISOString?.() ?? '')
                        //     }
                        // />
                        // <DatePicker
                        //     value={props.value ? new Date(props.value) : undefined}
                        //     onChange={date => handleValueChange(date?.toISOString?.() ?? '')}
                        // />
                        <DateTimeIsoPicker
                            placeholder={placeholder}
                            value={props.value ?? ''}
                            onChange={date => handleValueChange(date ?? '')}
                        />
                    );
                case 'tags':
                    return (
                        <Select
                            mode="tags"
                            value={props.value ? props.value.split(',') : []}
                            placeholder={placeholder}
                            onChange={arr =>
                                handleValueChange((arr as string[]).join(','))
                            }
                        />
                    );
                case 'enum':
                    const opt = props.value !== undefined ? props.field.enumOptions?.find?.(o => o.value === props.value) : undefined
                    return <Select
                        options={props.field.enumOptions ?? []}
                        value={opt?.value ?? null as string}
                        onChange={value => handleValueChange(value as string)}
                    />
                // return <>
                //     <Dropdown // fluentui
                //         value={opt?.value ?? null as string}
                //         selectedOptions={opt ? [opt.value] : []}
                //         onOptionSelect={props.onChange ? (event, data) => handleValueChange(data.optionValue) : undefined}
                //     >
                //         {props.field.enumOptions.map((opt, i) =>
                //             <Option key={`${i}-${opt.value}`} value={opt.value} text={opt.text ?? (typeof opt.label === 'string' ? opt.label : opt.value)}>
                //                 {opt.label}
                //             </Option>)}
                //     </Dropdown>
                // </>
                default:
                    return null;
            }
        }, [props.field.type, props.value, handleValueChange]);

        return <FieldDescription
            field={<Stack rowGap="XS">
                <ControlLabel>{props.field.label}</ControlLabel>
                {widget}
            </Stack>}
            description={props.field.description}
        />
    },
    // Custom compare: only re-render if the `value` actually changed
    (prev, next) => prev.value === next.value
);


const parseSimpleValidFloatOrNull: (x: string | number | undefined | null) => number | null = x => {
    try {
        const y = Number(x)
        if (Number.isNaN(y) || !Number.isFinite(y)) {
            return null
        }
        else {
            return y
        }
    }
    catch (e) {
        return null
    }
}