/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with INIshell.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The PreviewWindo shows the current INI file and offers to save it.
 * 2019-11
 */

#ifndef PREVIEWWINDOW_H
#define PREVIEWWINDOW_H

#include "src/main/INIParser.h"

#include <QAction>
#include <QLineEdit>
#include <QMainWindow>
#include <QToolButton>
#include <QRegularExpression>
#include <QString>
#include <QSyntaxHighlighter>
#include <QTabWidget>
#include <QTextDocument>
#include <QTextEdit>
#include <QVector>
#include <QWidget>

class SyntaxHighlighter : public QSyntaxHighlighter
{
	Q_OBJECT

	public:
		SyntaxHighlighter(QTextDocument *textdoc = nullptr);

	protected:
		void highlightBlock(const QString &text) override;

	private:
		struct HighlightingRule {
			QRegularExpression pattern;
			QTextCharFormat format;
		};
		QVector<HighlightingRule> rules_;
};

class PreviewWindow : public QMainWindow
{
	Q_OBJECT

	public:
		explicit PreviewWindow(QMainWindow *parent = nullptr);
		void addIniTab();
		int count() const { return file_tabs_->count(); }

	protected:
		void closeEvent(QCloseEvent* event) override;
		void keyPressEvent(QKeyEvent *event) override;

	private:
		enum transform_whitespaces { //options for transformations
			SINGLE_WS,
			LONGEST_WS
		};
		enum transform_capitalization {
			LOWER_CASE,
			UPPER_CASE
		};
		enum transform_comments {
			DELETE
		};

		void closeTab(int index);
		void createMenu();
		void createFindBar();
		void showFindBar();
		void hideFindBar();
		void textChanged(const int &index);
		void loadIniWithGui();
		void writeIniToFile(const QString &file_name);
		int warnOnUnsavedIni();
		void previewStatus(const QString &text);
		void setMonospacedFont(QTextEdit *editor);
		QTextEdit * getCurrentEditor();
		QString getCurrentFilename() const;
		void setTextWithHistory(QTextEdit *editor, const QString& text);
		void transformWhitespaces(const transform_whitespaces &mode);
		void transformCapitalization(const transform_capitalization &mode);
		void transformComments(const transform_comments &mode);

		INIParser preview_ini_; //our local INIParser to do transformations on
		QTabWidget *file_tabs_ = nullptr;
		SyntaxHighlighter *highlighter_ = nullptr;
		QLineEdit *find_text_ = nullptr;
		QToolButton *close_find_bar_ = nullptr;
		bool has_sorted_alphabetically_ = false;
		QAction *file_save_and_load_ = nullptr;
		QAction *file_load_ = nullptr;

	private slots:
		void saveFile();
		void saveFileAs();
		void saveFileAndLoadIntoGui();
		void loadIntoGui();
		void quickBackup();
		void onFindTextChanged(const QString &text);
		void onTransformMenuClick(const QString &action);
};

#endif //PREVIEWWINDOW_H
