/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "common.h"
#include "colors.h"
#include "src/gui/Settings.h"

#include <QDir>
#include <QFileInfo>

namespace html {

/**
 * @brief Add HTML embolden tags to text.
 * @param[in] text Text to add embolden tags to.
 * @return Text enclosed by embolden tags.
 */
QString bold(const QString &text)
{
	return "<b>" + text + "</b>";
}

/**
 * @brief Add HTML color tags to text.
 * @param[in] text Text to add color tags to.
 * @return Text enclosed by color tags.
 */
QString color(const QString &text, const QString &color)
{
	QString strRet( "<font color=\"" + colors::getQColor(color).name() + "\">" + text + "</font>");
	return strRet;
}

} //namespace html

/**
 * @brief Retrieve the message of an event that wants to communicate.
 * @details This function is used for example by the XML schema validation.
 * @param[in] type Type of the message (unused).
 * @param[in] description A text description of what has happened.
 * @param[in] identifier Identifier for the message (unused).
 * @param[in] location The location something has happened at (e. g. line number for text errors).
 */
void MessageHandler::handleMessage(QtMsgType type, const QString &description, const QUrl &identifier,
    const QSourceLocation &location)
{
	Q_UNUSED(type) //turn off compiler warnings
	Q_UNUSED(identifier)

	description_ = description;
	location_ = location;
}

/**
 * @brief Fill a list with directories to search for XMLs.
 * @details This function queries a couple of default folders on various systems, as well as
 * ones that can be set by the user. Duplicates (e. g. the same folder given by a relative
 * and an absolute path) are ignored.
 * @return A list of directories to search for XMLs.
 */
QStringList getSearchDirs(const bool &include_user_set, const bool &include_nonexistent_folders)
{
	QStringList dirs;
	//TODO: recursive search

	/* hardcoded directories */
	dirs << "./applications";
	dirs << "./simulations";
	dirs << QDir::currentPath();
	dirs << QCoreApplication::applicationDirPath();
	dirs << "/usr/share/inishell/applications";
	dirs << "/usr/share/inishell/simulations";
	dirs << "/opt/inishell/applications";
	dirs << "/opt/inishell/simulations";
	QStringList documents_folders(QStandardPaths::standardLocations(QStandardPaths::DocumentsLocation));
	for (auto doc_dir : documents_folders) {
		dirs << doc_dir + "/applications";
		dirs << doc_dir + "/simulations";
	}

	if (include_user_set) {
		/* user set paths */
		const QString user_applications_path(getSetting("auto::history::xml_applications_folder", "path"));
		if (!user_applications_path.isEmpty()) //can be set through a right-click on the applications list
			dirs << user_applications_path;
		const QString user_simulations_path(getSetting("auto::history::xml_simulations_folder", "path"));
		if (!user_simulations_path.isEmpty()) //via right-click on simulations
			dirs << user_simulations_path;
	}

	//now, check that we don't have the same folder multiple times (e. g. via relative and absolute paths):
	QStringList filtered_dirs;
	for (auto &dir : dirs) {
		QFileInfo dinfo(dir);
		if (!include_nonexistent_folders && !dinfo.exists())
			continue;
		bool found = false;
		for (auto filtered_dir : filtered_dirs) {
			QFileInfo fdinfo(filtered_dir);
			if (dinfo.absoluteFilePath() == fdinfo.absoluteFilePath()) {
				found = true;
				break;
			}
		}
		if (!found)
			filtered_dirs << dir;
	}

	return filtered_dirs;
}
