/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "TerminalView.h"
#include "src/main/inishell.h"

#include <QDir>
#include <QVBoxLayout>

/**
 * @brief Default constructor for a TerminalView.
 * @details This constructor creates a text box for terminal output.
 * @param[in] parent The TerminalView's parent window.
 */
TerminalView::TerminalView(QWidget *parent) : QWidget(parent)
{
	/* text box for terminal output */
	console_ = new QTextEdit;
	console_->setReadOnly(true);
	console_->setToolTip(tr("Text output of external programs"));
	log(html::color(html::bold("$ " + QDir::currentPath()), "sysinfo"));

	/* main layout */
	auto *main_layout = new QVBoxLayout;
	main_layout->addWidget(console_);
	this->setLayout(main_layout);
}

/**
 * @brief Log text to the terminal text box.
 * @param[in] text The text to log.
 * @param[in] is_std_err True if the text comes from stderr, false if from stdout.
 */
void TerminalView::log(const QString &text, const bool &is_std_err)
{
	//to prevent inserting new lines before all appends, it is necessary
	//to move the cursor to the end and insert the text:
	if (!is_std_err) { //output stems from stdout, i. e. all is well
		console_->moveCursor(QTextCursor::End);
		console_->insertHtml(text+"<br>");
		return;
	}

	if (!text.contains("\033[")) { //no extra styling given: use error color
		console_->moveCursor(QTextCursor::End);
		console_->insertHtml(html::color(html::bold(text+"<br>"), "error"));
	} else { //text styling requested from the output stream
		QString tmp( text ); //support terminal color-style
		tmp.replace("\033[01;30m", "<span style=\"color:LightGrey;\">");
		tmp.replace("\033[31;1m", "<span style=\"color:Red; font-style: bold;\">");
		tmp.replace("\033[4m", "<span style=\"text-decoration: underline;\">");
		tmp.replace("\033[3m", "<span style=\"color:LightGrey; font-style: italic;\">");
		//tmp.replace("\033[23m", "<span style=\"font-style: normal;\">");
		tmp.replace("\033[23m", "</span>");
		tmp.replace("\033[0m", "</span>");
		tmp.replace("\n", "<br>");

		console_->moveCursor(QTextCursor::End);
		console_->insertHtml(tmp+"<br>");
	}
}
