/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "PreviewWindow.h"
#include "src/gui/Settings.h"
#include "src/gui_elements/Atomic.h"
#include "src/main/colors.h"
#include "src/main/dimensions.h"
#include "src/main/inishell.h"

#include <QCoreApplication>
#include <QFile>
#include <QFileDialog>
#include <QMenuBar>
#include <QMessageBox>
#include <QStatusBar>

#ifdef DEBUG
	#include <iostream>
#endif //def DEBUG

/**
 * @class SyntaxHighlighter
 * @brief Default constructor for a syntax highlighter used in the INI preview.
 * @param[in] textdoc Text document to handle syntax highlighting for.
 */
SyntaxHighlighter::SyntaxHighlighter(QTextDocument *textdoc) : QSyntaxHighlighter(textdoc)
{
	HighlightingRule rule;

	/*
	 * The SyntaxHighlighter parses a document for INI file syntax and checks the sections
	 * and keys against the curently loaded XML. Sections and keys that are not present
	 * in the XML are highlighted differently.
	 */

	/* unknown sections */
	QTextCharFormat format_section;
	format_section.setForeground(colors::getQColor("syntax_unknown_section"));
	format_section.setFontWeight(QFont::Bold);
	rule.pattern = QRegularExpression(R"(.*\)" + Cst::section_open + R"(.*\)" + Cst::section_close + R"(.*)");
	rule.format = format_section;
	rules_.append(rule);

	/* unknown keys */
	QTextCharFormat format_unknown_key;
	format_unknown_key.setForeground(colors::getQColor("syntax_unknown_key"));
	rule.pattern = QRegularExpression(R"(^\s*\w+(?=\s*=))");
	rule.format = format_unknown_key;
	rules_.append(rule);

	/* INI values */
	QTextCharFormat format_value;
	format_value.setForeground(colors::getQColor("syntax_value"));
	rule.pattern = QRegularExpression(R"((?<=\=).*)");
	rule.format = format_value;
	rules_.append(rule);

	/* populate highlighter with known sections and keys */
	QTextCharFormat format_known_key;
	format_known_key.setForeground(colors::getQColor("syntax_known_key"));
	QTextCharFormat format_known_section;
	format_known_section.setForeground(colors::getQColor("syntax_known_section"));
	format_known_section.setFontWeight(QFont::Bold);

	QList<Atomic *> panel_list = getMainWindow()->findChildren<Atomic *>();
	for (auto &panel : panel_list) {
		if (panel->property("no_ini").toBool() == true) //e. g. Groups / frames
			continue;
		QString value, section, key;
		value = panel->getIniValue(section, key);
		rule.pattern = QRegularExpression("\\" + Cst::section_open + section + "\\" +
		    Cst::section_close, QRegularExpression::CaseInsensitiveOption); //TODO: escape only if needed for the set char
		rule.format = format_known_section;
		rules_.append(rule);
		rule.pattern = QRegularExpression(R"(^\s*)" + key + R"((=|\s))", QRegularExpression::CaseInsensitiveOption);
		rule.format = format_known_key;
		rules_.append(rule);
	}

	/* comments */
	QTextCharFormat format_block_comment;
	format_block_comment.setForeground(colors::getQColor("syntax_comment"));
	rule.pattern = QRegularExpression(R"(^\s*[#;].*)");
	rule.format = format_block_comment;
	rules_.append(rule);
	rule.pattern = QRegularExpression(R"(([#;](?!$).*))");
	rules_.append(rule);

	/* equals sign */
	QTextCharFormat format_equal;
	format_equal.setForeground(Qt::black);
	rule.pattern = QRegularExpression(R"(=)");
	rule.format = format_equal;
	rules_.append(rule);
}

/**
 * @brief Send a chunk of text through the syntax highlighter.
 * @param[in] text The text to syntax-highlight.
 */
void SyntaxHighlighter::highlightBlock(const QString &text)
{
	for (const HighlightingRule &rule : qAsConst(rules_)) {
		QRegularExpressionMatchIterator mit = rule.pattern.globalMatch(text);
		while (mit.hasNext()) { //run trough regex matches and set the stored formats
			QRegularExpressionMatch match = mit.next();
			setFormat(match.capturedStart(), match.capturedLength(), rule.format);
		}
	}
}

/**
 * @class PreviewWindow
 * @brief Default constructor for the PreviewWindow.
 * @details This constructor creates a tab bar to show multiple INI file versions in.
 * @param [in] parent The PreviewWindow's parent window (the main window).
 */
PreviewWindow::PreviewWindow(QMainWindow *parent) : QMainWindow(parent)
{
	file_tabs_ = new QTabWidget;
	connect(file_tabs_, &QTabWidget::tabCloseRequested, this, &PreviewWindow::closeTab);
	file_tabs_->setTabsClosable(true);
	this->setCentralWidget(file_tabs_);
	createMenu();

	dim::setDimensions(this, dim::PREVIEW); //size this window
	this->setWindowTitle(tr("Preview") + " ~ " + QCoreApplication::applicationName());
}

/**
 * @brief Display the current INI file in a new tab.
 */
void PreviewWindow::addIniTab()
{
	/* get currently set INI values */
	QString ini_contents;
	QTextStream ss(&ini_contents);
	INIParser gui_ini = getMainWindow()->getIniCopy();
	getMainWindow()->getControlPanel()->setIniValuesFromGui(&gui_ini);

	/* text box for the current INI */
	auto *preview_editor = new QTextEdit;
	preview_editor->setStyleSheet("QTextEdit {background-color: " + colors::getQColor("syntax_background").name() +
	    "; color: " + colors::getQColor("syntax_invalid").name() + "}");
	highlighter_ = new SyntaxHighlighter(preview_editor->document());

	/* display the current GUI's contents */
	gui_ini.outputIni(ss);
	if (ini_contents.isEmpty()) {
		ini_contents = tr("#Empty INI file\n") +
		tr("#Either all values are at their defaults (in which case they will not be written out), or you haven't opened an application and/or INI file yet.");
	}
	preview_editor->setPlainText(ini_contents);
	QFileInfo file_info(getMainWindow()->getIni()->getFilename());
	QString file_name(file_info.fileName());
	QString file_path(file_info.filePath());
	if (file_name.isEmpty()) //pick file name if no INI is opened yet
		file_name = "io_new.ini";
	else
		file_name += " *"; //asterisk for "not saved yet", unless it's only the info text
	if (file_path.isEmpty())
		file_path = QDir::currentPath();
	file_tabs_->addTab(preview_editor, file_name);
	file_tabs_->setTabToolTip(file_tabs_->count() - 1, file_path);
	file_tabs_->setCurrentIndex(file_tabs_->count() - 1); //switch to new tab
	connect(preview_editor, &QTextEdit::textChanged, this, [=]{ textChanged(file_tabs_->count() - 1); });

}

/**
 * @brief Event listener for when the window is being closed.
 * @details This function allows the user to cancel the closing if there are unsaved
 * changes in the opened INI files.
 * @param[in] event The close event.
 */
void PreviewWindow::closeEvent(QCloseEvent *event)
{
	bool has_unsaved_changes = false; //run through tabs and check for asterisks
	for (int ii = 0; ii < file_tabs_->count(); ++ii) {
		if (file_tabs_->tabText(ii).endsWith("*")) {
			has_unsaved_changes = true;
			break;
		}
	}
	if (has_unsaved_changes) { //at least one tab has unsaved changes
		const int cancel = warnOnUnsavedIni();
		if (cancel == QMessageBox::Cancel) {
			event->ignore();
			return;
		}
	}
	event->accept();
}


/**
 * @brief Event listener for the ESC key.
 * @details Close the Preview on pressing ESC.
 * @param event The key press event that is received.
 */
void PreviewWindow::keyPressEvent(QKeyEvent *event)
{
	if (event->key() == Qt::Key_Escape)
		this->close();
}

/**
 * @brief Close an INI file's tab.
 * @param[in] index Index of the tab that is being closed.
 */
void PreviewWindow::closeTab(int index)
{
	//Check for unsaved changes. Note that changes that cancel out leaving the INI file
	//unaltered will strill trigger the warning.
	if (file_tabs_->tabText(index).endsWith("*")) { //not saved yet
		const int cancel = warnOnUnsavedIni();
		if (cancel == QMessageBox::Cancel)
			return;
	}

	file_tabs_->removeTab(index);
	if (file_tabs_->count() == 0)
		this->close();
}

/**
 * @brief Create the PreviewWindow's menu.
 */
void PreviewWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_save = new QAction(tr("&Save"));
	file_save->setShortcut(Qt::CTRL + Qt::Key_S);
	menu_file->addAction(file_save);
	connect(file_save, &QAction::triggered, this, &PreviewWindow::saveFile);
	auto *file_save_as = new QAction(tr("Save &as..."));
	file_save_as->setShortcut(Qt::SHIFT + Qt::CTRL + Qt::Key_S);
	menu_file->addAction(file_save_as);
	connect(file_save_as, &QAction::triggered, this, &PreviewWindow::saveFileAs);
}

/**
 * @brief Delegated event listener for when the preview text changes.
 * @details A lambda calls this function with the text box index to add an asterisk
 * to the title ("not saved").
 * @param[in] index Index of the tab the text changed in.
 */
void PreviewWindow::textChanged(const int &index)
{
	const QString file_name(file_tabs_->tabText(index));
	if (!file_name.endsWith("*"))
		file_tabs_->setTabText(index, file_name + " *");

}

/**
 * @brief Write an opened INI file to the file system.
 * @param[in] file_name The path to save to.
 */
void PreviewWindow::writeIniToFile(const QString &file_name)
{
	QFile outfile(file_name);
	if (!outfile.open(QIODevice::WriteOnly)) {
		statusBar()->showMessage(tr("Could not open ") + file_name); //temporary message
		return;
	}
	QTextStream ss(&outfile);
	auto text_box = qobject_cast<QTextEdit *>(file_tabs_->widget(file_tabs_->currentIndex()));
	if (text_box)
		ss << text_box->toPlainText();
	outfile.close();

	QFileInfo finfo(file_name); //switch the displayed name to new file (without askterisk)
	QString shown_name(finfo.fileName());
	file_tabs_->setTabText(file_tabs_->currentIndex(), shown_name);
	statusBar()->showMessage(tr("Saved to ") + file_name);

	setSetting("auto::history::last_preview_write", "path", finfo.absoluteDir().path());
}

/**
 * @brief Ask a user if they want to cancel an action.
 * @details This function is called before closing INI tabs with unsaved changes.
 * @return Button the user has clicked.
 */
int PreviewWindow::warnOnUnsavedIni()
{
	QMessageBox msgNotSaved;
	msgNotSaved.setWindowTitle("Warning ~ " + QCoreApplication::applicationName());
	msgNotSaved.setText(tr("<b>INI file not saved yet.</b>"));
	msgNotSaved.setInformativeText(tr("Your INI file(s) may contain unsaved changes."));
	msgNotSaved.setIcon(QMessageBox::Warning);
	msgNotSaved.setStandardButtons(QMessageBox::Cancel | QMessageBox::Discard);
	msgNotSaved.setDefaultButton(QMessageBox::Cancel);
	int clicked = msgNotSaved.exec();
	return clicked;
}

/**
 * @brief Save an INI file to the file system.
 */
void PreviewWindow::saveFile()
{
	QString shown_name(file_tabs_->tabText(file_tabs_->currentIndex()));
	if (shown_name.endsWith("*")) //remove "unsaved" asterisk for the file name
		shown_name.chop(2);
	const QString file_name(file_tabs_->tabToolTip(file_tabs_->currentIndex()) +
	    "/" + shown_name);
	writeIniToFile(file_name);
}

/**
 * @brief Save an INI file to the file system.
 */
void PreviewWindow::saveFileAs()
{
	QString start_path(getSetting("auto::history::last_preview_write", "path"));
	if (start_path.isEmpty())
		start_path = QDir::currentPath();
	QString proposed_name(file_tabs_->tabText(file_tabs_->currentIndex()));
	if (proposed_name.endsWith("*"))
		proposed_name.chop(2);
	const QString file_name = QFileDialog::getSaveFileName(this, tr("Save INI file"),
	    start_path + "/" + proposed_name, "INI files (*.ini *.INI);;All files (*)",
	    nullptr, QFileDialog::DontUseNativeDialog);
	if (file_name.isNull()) //cancelled
		return;
	writeIniToFile(file_name);
}
