/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The main window that is displayed on startup.
 * 2019-10
 */

#include "MainWindow.h"
#include "src/gui/AboutWindow.h"
#include "src/gui/Settings.h"
#include "src/gui_elements/Group.h" //to exclude Groups from panel search
#include "src/main/colors.h"
#include "src/main/Error.h"
#include "src/main/dimensions.h"
#include "src/main/INIParser.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QApplication>
#include <QCheckBox>
#include <QDesktopServices>
#include <QFileDialog>
#include <QFileInfo>
#include <QGroupBox>
#include <QMenuBar>
#include <QMessageBox>
#include <QRegularExpression>
#include <QSpacerItem>
#include <QStatusBar>
#include <QSysInfo>
#include <QToolBar>

#ifdef DEBUG
	#include <iostream>
#endif

/**
 * @class MouseEventFilter
 * @brief Mouse event listener for the main program window.
 * @param[in] object Object the event stems from.
 * @param[in] event The type of event.
 * @return True if the event was accepted.
 */
bool MouseEventFilter::eventFilter(QObject *object, QEvent *event)
{
	if (event->type() == QEvent::MouseButtonPress) {
		if (object->property("mouseclick") == "open_log") {
			getMainWindow()->viewLogger();
		} else if (object->property("mouseclick") == "open_ini") {
			if (!getMainWindow()->ini_filename_->text().isEmpty())
				QDesktopServices::openUrl("file:" + getMainWindow()->ini_filename_->text());
		}
	}
	return QObject::eventFilter(object, event); //pass to actual event of the object
}

/**
 * @class MainWindow
 * @brief Constructor for the main window.
 * @param[in] xml_settings Settings for the static INIshell GUI.
 * @param[in] parent Unused since the main window is kept in scope by the entry point function.
 */
MainWindow::MainWindow(QDomDocument &xml_settings, QString &settings_location, QStringList errors, QMainWindow *parent)
    : QMainWindow(parent), logger_(this), xml_settings_(xml_settings), xml_settings_filename_(settings_location)
{

	//TODO: Sketch the program workflow.

	status_timer_ = new QTimer; //for temporary status messages
	status_timer_->setSingleShot(true);
	connect(status_timer_, &QTimer::timeout, this, &MainWindow::clearStatus);

	logger_.logSystemInfo();
	for (auto &err : errors) //errors from main.cc before the Logger existed
		logger_.log(err, "error");

	/* retrieve and set main window geometry */
	dim::setDimensions(this, dim::MAIN_WINDOW);
	dim::setDimensions(&logger_, dim::LOGGER);

	/* create basic GUI items */
	createMenu();
	createToolbar();
	createStatusbar();

	preview_ = new PreviewWindow(this);
	/* create the dynamic GUI area */
	control_panel_ = new MainPanel;
	this->setCentralWidget(control_panel_);
	ini_.setLogger(&logger_);
	setStatus("Ready.", "sysinfo");

	//TODO: Auto-open last XML for people who want to keep the workflow panel collapsed? Maybe as a setting?
}

/**
 * @brief Destructor for the main window.
 * @details This function is called after all safety checks have already been performed.
 */
MainWindow::~MainWindow()
{ //safety checks are performed in closeEvent()

	setWindowSizeSettings(); //put the main window sizes into the settings XML
	saveSettings(xml_settings_);
}

/**
 * @brief Build the dynamic GUI.
 * This function initiates the recursive GUI building with an XML document that was
 * parsed beforehand.
 * @param[in] xml XML to build the gui from.
 */
void MainWindow::buildGui(const QDomDocument &xml)
{
	setUpdatesEnabled(false); //disable painting until done
	QDomNode root = xml.firstChild();
	while (!root.isNull()) {
		if (root.isElement()) { //skip over comments
			//give no parent group - tabs will be created for top level:
			recursiveBuild(root, nullptr, QString());
			break;
		}
		root = root.nextSibling();
	}
	setUpdatesEnabled(true);
}

/**
 * @brief Set the loaded panels' values from an INI file.
 * @param[in] ini The INI file in form of an INIParser (usually the main one).
 * @return
 */
void MainWindow::setGuiFromIni(const INIParser &ini)
{
	for (auto &sec : ini.getSections()) { //run through sections in INI file
		ScrollPanel *tab_scroll = getControlPanel()->getSectionScrollarea(sec.getName(),
		    QString(), QString(), true); //get the corresponding tab of our GUI
		if (tab_scroll != nullptr) { //section exists in GUI
			for (auto &kv : sec.getKeyValueList() ) {
				//find the corresponding panel, and try to create it for dynamic panels
				//(e. g. Selector, Replicator):
				QWidgetList widgets = findPanel(tab_scroll, sec, kv.second);
				if (!widgets.isEmpty()) {
					for (int ii = 0; ii < widgets.size(); ++ii) //multiple panels can share the same key
						widgets.at(ii)->setProperty("ini_value", kv.second.getValue());
				} else {
					logger_.log(tr("No GUI element found for INI key \"") + sec.getName() +
					    Cst::sep + kv.second.getKey() + "\"", "iniwarning");
					topStatus(tr("Encountered unknown INI key"), "iniwarning");
				}
			} //endfor kv
		} else { //section does not exist
			log(tr("No matching section in GUI for section from INI file: \"") +
			    sec.getName() + "\"", "iniwarning");
			topStatus(tr("Encountered unknown INI key"), "iniwarning");
		} //endif section exists
	} //endfor sec
}

/**
 * @brief Retrieve all panels that handle a certain INI key.
 * @param[in] ini_key The INI key to find.
 * @return A list of all panels that handle the INI key.
 */
QList<Atomic *> MainWindow::getPanelsForKey(const QString &ini_key)
{
	QList<Atomic *> panel_list = control_panel_->findChildren<Atomic *>(Atomic::getQtKey(ini_key));
	for (auto it = panel_list.begin(); it != panel_list.end(); ++it) {
		//groups don't count towards finding INI keys (for this reason, they additionally
		//have the "no_ini" key set):
		if (qobject_cast<Group *>(*it))
			panel_list.erase(it);
	}
	return panel_list;
}

/**
 * @brief Save the current GUI to an INI file.
 * @details This function calls the underlying function that does this and displays a message if
 * the user has neglected to set some mandatory INI values.
 * @param[in] filename The file to save to. If not given, the current INI file will be used.
 */
void MainWindow::saveIni(const QString &filename)
{
	/*
	 * We keep the original INIParser as-is, i. e. we always keep the INI file as it was loaded
	 * originally. This is solemnly to be able to check if anything has changed through user
	 * interaction, and if yes, warn before closing.
	 */
	INIParser gui_ini = ini_; //an INIParser is safe to copy around - this is a deep copy
	const QString missing = control_panel_->setIniValuesFromGui(&gui_ini);

	if (!missing.isEmpty()) {
		QMessageBox msgMissing;
		msgMissing.setWindowTitle("Warning ~ " + QCoreApplication::applicationName());
		msgMissing.setText(tr("<b>Missing mandatory INI values.</b>"));
		msgMissing.setInformativeText(tr("Some non-optional INI keys are not set.\nSee details for a list or go back to the GUI and set all highlighted fields."));
		msgMissing.setDetailedText(tr("Missing INI keys: \n") + missing);
		msgMissing.setIcon(QMessageBox::Warning);
		msgMissing.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
		msgMissing.setDefaultButton(QMessageBox::Cancel);
		int clicked = msgMissing.exec();
		if (clicked == QMessageBox::Cancel)
			return;
	}
	//if no file is specified we save to the currently open INI file (save vs. save as):
	gui_ini.writeIni(filename.isEmpty()? gui_ini.getFilename() : filename);
}

/**
 * @brief Save the currently set values to a new INI file.
 */
void MainWindow::saveIniAs()
{
	QString start_path(getSetting("auto::history::last_preview_write", "path"));
	if (start_path.isEmpty())
		start_path = QDir::currentPath();

	const QString filename = QFileDialog::getSaveFileName(this, tr("Save INI file"),
	    start_path + "/" + ini_filename_->text(),
	    "INI files (*.ini *.INI);;All files (*)", nullptr, QFileDialog::DontUseNativeDialog);
	if (filename.isNull()) //cancelled
		return;
	saveIni(filename);
	ini_.setFilename(filename); //the new file is the new current file like in all programs
	ini_filename_->setText(filename);
	autoload_->setVisible(true); //if started from an empty GUI, this could still be disabled
	toolbar_save_ini_->setEnabled(true); //toolbar entry
	file_save_ini_->setEnabled(true); //menu entry

	const QFileInfo finfo(filename);
	setSetting("auto::history::last_ini", "path", finfo.absoluteDir().path());
}

/**
 * @brief Select a path for an INI file to be opened, and then open it.
 */
void MainWindow::openIni()
{
	QString start_path(getSetting("auto::history::last_ini", "path"));
	if (start_path.isEmpty())
		start_path = QDir::currentPath();

	const QString path = QFileDialog::getOpenFileName(this, tr("Open INI file"), start_path,
	    "INI files (*.ini);;All files (*)", nullptr, QFileDialog::DontUseNativeDialog | QFileDialog::DontConfirmOverwrite);
	if (path.isNull()) //cancelled
		return;
	openIni(path);

	const QFileInfo finfo(path);
	setSetting("auto::history::last_ini", "path", finfo.absoluteDir().path());
}

/**
 * @brief Open an INI file and set the GUI to it's values.
 * @param[in] path The INI file to open.
 * @param[in] is_autoopen Is the INI being loaded through the autoopen mechanism?
 */
void MainWindow::openIni(const QString &path, const bool &is_autoopen, const bool &fresh)
{
	topStatus(tr("Reading INI file..."), "black", true);
	refreshStatus();
	if (fresh)
		clearGui();
	ini_.setFile(path); //load the file into the main INI parser
	setGuiFromIni(ini_); //set the GUI to the INI file's values

	toolbar_save_ini_->setEnabled(true);
	file_save_ini_->setEnabled(true);
	autoload_->setVisible(true);
	ini_filename_->setText(path);
	autoload_box_->setText(tr("autoload this INI for ") + current_application_);
	if (!is_autoopen) //when a user clicks an INI file to open it we ask anew whether to autoopen
		autoload_box_->setCheckState(Qt::Unchecked);
	topStatus(tr("INI file ready."), "black", false);
}

/**
 * @brief Close the currently opened INI file.
 * @details This function checks whether the user has made changes to the INI file and if yes,
 * allows to save first, discard changes, or cancel the operation.
 * @return True if the INI file was closed, false if the user has cancelled.
 */
bool MainWindow::closeIni()
{
	if (!help_loaded_) { //user currently has the help opened which we always allow to close
		/*
		 * We leave the original INIParser - the one that holds the values like they were
		 * originally read from an INI file - intact and make a copy of it. The currently
		 * set values from the GUI are loaded into the copy, and then the two are compared
		 * for changes. This way we don't display a "settings may be lost" warning if in
		 * fact nothing has changed, resp. the changes cancelled out.
		 */
		INIParser gui_ini = ini_;
		(void) control_panel_->setIniValuesFromGui(&gui_ini);
		if (ini_ != gui_ini) {
			QMessageBox msgNotSaved;
			msgNotSaved.setWindowTitle("Warning ~ " + QCoreApplication::applicationName());
			msgNotSaved.setText(tr("<b>INI settings will be lost.</b>"));
			msgNotSaved.setInformativeText(tr("Some INI keys you have set will be lost if you don't save the current INI file."));
			msgNotSaved.setIcon(QMessageBox::Warning);
			msgNotSaved.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel | QMessageBox::Discard);
			msgNotSaved.setDefaultButton(QMessageBox::Cancel);
			int clicked = msgNotSaved.exec();
			if (clicked == QMessageBox::Cancel)
				return false;
			else if (clicked == QMessageBox::Save)
				saveIni();
		}
	} //endif help_loaded

	ini_.clear();
	toolbar_save_ini_->setEnabled(false);
	file_save_ini_->setEnabled(false);
	ini_filename_->setText(QString());
	autoload_->setVisible(false);
	return true;
}

/**
 * @brief Reset the GUI to default values.
 * @details This function checks whether there were any changes to the INI values
 * (if yes, the user can save first or discard or cancel), then clears the whole GUI.
 */
void MainWindow::clearGui()
{
	const bool perform_close = closeIni();
	if (!perform_close) //user clicked 'cancel'
		return;
	getControlPanel()->clearGui();
	ini_filename_->setText(QString());
	autoload_->setVisible(false);
}

/**
 * @brief Store the main window sizes in the settings XML.
 */
void MainWindow::setWindowSizeSettings()
{
	QList<int> splitter_sizes(control_panel_->getSplitterSizes());
	setSetting("auto::sizes::splitter_workflow", "size", QString::number(splitter_sizes.at(0)));
	setSetting("auto::sizes::splitter_mainpanel", "size", QString::number(splitter_sizes.at(1)));

	setSetting("auto::sizes::window_" + QString::number(dim::window_type::PREVIEW),
	    "width", QString::number(preview_->width()));
	setSetting("auto::sizes::window_" + QString::number(dim::window_type::PREVIEW),
	    "height", QString::number(preview_->height()));
	setSetting("auto::sizes::window_" + QString::number(dim::window_type::LOGGER),
	    "width", QString::number(logger_.width()));
	setSetting("auto::sizes::window_" + QString::number(dim::window_type::LOGGER),
	    "height", QString::number(logger_.height()));
	setSetting("auto::sizes::window_" + QString::number(dim::window_type::MAIN_WINDOW),
	    "width", QString::number(this->width()));
	setSetting("auto::sizes::window_" + QString::number(dim::window_type::MAIN_WINDOW),
	    "height", QString::number(this->height()));

	//remember the toolbar position:
	setSetting("auto::position::toolbar", "position", QString::number(this->toolBarArea(toolbar_)));
}

/**
 * @brief Open an XML file containing an application/simulation.
 * @param[in] path Path to the file to open.
 * @param[in] app_name When looking for applications, the application name was parsed and put
 * into a list. This is the app_name so we don't have to parse again.
 * @param[in] fresh If true, reset the GUI before opening a new application.
 */
void MainWindow::openXml(const QString &path, const QString &app_name, const bool &fresh)
{
	if (fresh) {
		const bool perform_close = closeIni();
		if (!perform_close)
			return;
		control_panel_->clearGuiElements();
		help_loaded_ = false;
	}

	if (QFile::exists(path)) {
		XMLReader xml;
		QString xml_error = QString();
		xml.read(path, xml_error);
		if (!xml_error.isNull()) {
			xml_error.chop(1); //trailing \n
			Error(tr("Errors occured when parsing the XML configuration file"),
			    tr("File: \"") + path + "\"", xml_error);
		}
		setStatus("Building GUI...", "black", true);
		buildGui(xml.getXml());
		setStatus("Ready.", "sysinfo", false);
		control_panel_->getWorkflowPanel()->buildWorkflowPanel(xml.getXml());
	} else {
		topLog(tr("An application or simulation file that has previously been found is now missing. Right-click the list to refresh."),
		     "xmlerror");
		topStatus(tr("File has been removed"));
		return;
	}

	toolbar_save_ini_as_->setEnabled(true);
	file_save_ini_as_->setEnabled(true);
	current_application_ = app_name;

	//run through all INIs that were saved to be autoloaded and check if it's the application we are opening:
	for (auto ini_node = xml_settings_.firstChildElement().firstChildElement("user").firstChildElement(
	    "autoload").firstChildElement("ini");
	    !ini_node.isNull(); ini_node = ini_node.nextSiblingElement("ini")) {
		if (ini_node.attribute("application").toLower() == app_name.toLower()) {
			autoload_box_->blockSignals(true); //don't re-save the setting we just fetched
			autoload_box_->setCheckState(Qt::Checked);
			autoload_box_->setText(tr("autoload this INI for ") + current_application_);
			autoload_box_->blockSignals(false);
			openIni(ini_node.text(), true); //TODO: delete if non-existent
			break;
		}
	}
	toolbar_clear_gui_->setEnabled(true);
	gui_clear_->setEnabled(true);
	toolbar_open_ini_->setEnabled(true); //toolbar entry
	file_open_ini_->setEnabled(true); //menu entry
	auto *file_view = getControlPanel()->getWorkflowPanel()->getFilesystemView();
	file_view->setEnabled(true);
	auto *path_label = file_view->getInfoLabel();
	path_label->setText(file_view->getInfoLabel()->property("path").toString());
	path_label->setWordWrap(false);
	path_label->setStyleSheet("QLabel {background-color: white}");
}

/**
 * @brief Find the panels that control a certain key/value pair.
 * @details This function includes panels that could be created by dynamic panels (Selector etc.).
 * @param[in] parent The parent panel or window to search, since usually we have some information
 * about where to find it. This function is used when iterating through INIParser elements
 * to load the GUI with values.
 * @param[in] section The section to find.
 * @param[in] keyval The key/value pair to find.
 * @return A list of all panels controlling the key/value pair.
 */
QWidgetList MainWindow::findPanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QWidgetList panels;
	//first, check if there is a panel for it loaded and available:
	panels = findSimplePanel(parent, section, keyval);
	//if not found, check if one of our Selectors can create it:
	int panel_count = parent->findChildren<Atomic *>().size();
	if (panels.isEmpty())
		panels = prepareSelector(parent, section, keyval);
	//if still not found, check if one of our Replicators can create it:
	if (panels.isEmpty())
		panels = prepareReplicator(parent, section, keyval);
	if (parent->findChildren<Atomic *>().size() != panel_count) //new elements were created
		return findPanel(parent, section, keyval); //recursion through Replicators that create Selectors that...
	else
		return panels; //no more suitable dynamic panels found
}

/**
 * @brief Find the panels that control a certain key/value pair.
 * @details This function does not include dynamic panels (Selector etc.) and searches only
 * what has already been created.
 * @param[in] parent The parent panel or window to search, because usually we have some information.
 * @param[in] section Section to find.
 * @param[in] keyval Key-value pair to find.
 * @return A list of all panels controlling the key/value pair.
 */
QWidgetList MainWindow::findSimplePanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	/* simple, not nested, keys */
	return parent->findChildren<QWidget *>(Atomic::getQtKey(id));
}

/**
 * @brief Find a Selector in our GUI that could handle a given INI key.
 * @details This function looks for Selectors handling keys like "%::COPY". If one is found,
 * it requests the Selector to create a new panel for the key which will then be available
 * to the parent function that is currently looking for a suitable panel for the INI key.
 * @param[in] parent The parent panel or window to search.
 * @param[in] section Section to find.
 * @param[in] keyval Key/value pair to find.
 * @return A list of panels for the INI key including the possibly newly created ones.
 */
QWidgetList MainWindow::prepareSelector(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	//INI key as it would be given in the file, i. e. with the parameters in place instead of the Selector's %:
	const QString regex_selector("^" + section.getName() + Cst::sep + R"((\w+)()" + Cst::sep + R"()(\w+?)([0-9]*$))");
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.captured(0) == id) { //it could be from a selector with template children
		static const size_t idx_parameter = 1;
		static const size_t idx_keyname = 3;
		static const size_t idx_optional_number = 4;
		const QString parameter(matches.captured(idx_parameter));
		const QString key_name(matches.captured(idx_keyname));
		const QString number(matches.captured(idx_optional_number));
		//this would be the ID given in an XML, i. e. with a "%" staing for a parameter:
		QString gui_id = section.getName() + Cst::sep + "%" + Cst::sep + key_name + (number.isEmpty()? "" : "#");

		//try to find Selector:
		QList<Selector *> selector_list = parent->findChildren<Selector *>(Atomic::getQtKey(gui_id));
		for (int ii = 0; ii < selector_list.size(); ++ii)
			selector_list.at(ii)->setProperty("ini_value", parameter); //cf. notes in prepareReplicator
		//now all the right child widgets should exist and we can try to find again:
		const QString key_id = section.getName() + Cst::sep + keyval.getKey(); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));
	}
	return QWidgetList();
}

/**
 * @brief Find a Replicator in our GUI that could handle a given INI key.
 * @details This function looks for Replicators handling keys like "STATION#". If one is found,
 * it requests the Replicator to create a new panel for the key which will then be available
 * to the parent function that is currently looking for a suitable panel for the INI key.
 * @param[in] parent The parent panel or window to search.
 * @param[in] section Section to find.
 * @param[in] keyval Key/value pair to find.
 * @return A list of panels for the INI key including the possibly newly created ones.
 */
QWidgetList MainWindow::prepareReplicator(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	//the key how it would look in an INI file:
	const QString regex_selector("^" + section.getName() + Cst::sep +
	    R"((\w+)" + Cst::sep + R"()*(\w*?)(?=\d)(\d*)$)");
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.captured(0) == id) { //it's from a replicator with template children
		/*
		 * If we arrive here, we try to find the panel for an INI key that could belong
		 * to the child of a Replicator, e. g. "INPUT::STATION1" or "FILTERS::TA::FILTER1".
		 * We extract the name the Replicator would have (number to "#") and try finding it.
		 */
		static const size_t idx_parameter = 1;
		static const size_t idx_key = 2;
		static const size_t idx_number = 3;
		QString parameter(matches.captured(idx_parameter)); //has trailing "::" if existent
		const QString key(matches.captured(idx_key));
		const QString number(matches.captured(idx_number));
		//they key how it would look in an XML:
		QString gui_id = section.getName() + Cst::sep + parameter + key + "#"; //Replicator's name

		/*
		 * A replicator can't normally be accessed via INI keys, because there is no
		 * standalone XML code for it. It always occurs together with child panels,
		 * and those are the ones that will be sought by the INI parser to set values.
		 * Therefore we can use the "ini_value" property listener for a Replicator to
		 * tell it to replicate its panel, thus creating the necessary child panels.
		 */
		QList<Replicator *> replicator_list = parent-> //try to find replicator:
		    findChildren<Replicator *>(Atomic::getQtKey(gui_id));
		for (int ii = 0; ii < replicator_list.count(); ++ii)
			replicator_list.at(ii)->setProperty("ini_value", number);
		//now all the right child panels should exist and we can try to find again:
		const QString key_id(section.getName() + Cst::sep + keyval.getKey()); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));
	} //endif has match

	return QWidgetList(); //no suitable Replicator found
}

/**
 * @brief Build the main window's menu items.
 */
void MainWindow::createMenu()
{
	this->menuBar()->setNativeMenuBar(false); //necessary to display menu on MacOS

	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	file_open_ini_ = new QAction(QIcon(":icons/ini_file.png"), tr("&Open INI file..."));
	menu_file->addAction(file_open_ini_);
	connect(file_open_ini_, &QAction::triggered, this, [=]{ toolbarClick("open_ini"); });
	file_open_ini_->setShortcut(Qt::CTRL + Qt::Key_O);
	file_save_ini_ = new QAction(QIcon(":icons/save.png"), tr("&Save INI file"));
	file_save_ini_->setShortcut(Qt::CTRL + Qt::Key_S);
	menu_file->addAction(file_save_ini_);
	file_save_ini_->setEnabled(false); //enable after an INI is opened
	connect(file_save_ini_, &QAction::triggered, this, [=]{ toolbarClick("save_ini"); });
	file_save_ini_as_ = new QAction(QIcon(":icons/save_as.png"), tr("Save INI file &as..."));
	file_save_ini_as_->setShortcut(Qt::SHIFT + Qt::CTRL + Qt::Key_S);
	menu_file->addAction(file_save_ini_as_);
	file_save_ini_as_->setEnabled(false);
	connect(file_save_ini_as_, &QAction::triggered, this, [=]{ toolbarClick("save_ini_as"); });
//	menu_file->addSeparator();
//	auto *file_open_xml = new QAction(QIcon(":icons/xml_file.png"), tr("Open &XML file..."));
//	menu_file->addAction(file_open_xml);
//	connect(file_open_xml, &QAction::triggered, this, &MainWindow::onMenuOpenXml);

	menu_file->addSeparator();
	auto *file_quit = new QAction(tr("&Exit"));
	file_quit->setShortcuts(QList<QKeySequence>() << (Qt::CTRL + Qt::Key_W) << (Qt::CTRL + Qt::Key_Q));
	menu_file->addAction(file_quit);
	connect(file_quit, &QAction::triggered, this, &MainWindow::quitProgram);

	/* GUI menu */
	QMenu *menu_gui = this->menuBar()->addMenu(tr("&GUI"));
	auto *gui_reset = new QAction(tr("&Reset GUI"));
	menu_gui->addAction(gui_reset);
	connect(gui_reset, &QAction::triggered, this, &MainWindow::resetGui);
	gui_reset->setShortcut(Qt::CTRL + Qt::SHIFT + Qt::Key_Backspace);
	gui_clear_ = new QAction(QIcon(":icons/clear.png"), tr("&Clear GUI"));
	menu_gui->addAction(gui_clear_);
	gui_clear_->setEnabled(false);
	gui_clear_->setShortcut(Qt::CTRL + Qt::Key_Backspace);
	connect(gui_clear_, &QAction::triggered, this, [=]{ toolbarClick("clear_gui"); });

	/* View menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	auto *view_preview = new QAction(QIcon(":icons/preview"), tr("P&review"));
	menu_view->addAction(view_preview);
	view_preview->setShortcut(Qt::CTRL + Qt::Key_R);
	connect(view_preview, &QAction::triggered, this, &MainWindow::viewPreview);
	auto *view_log = new QAction(tr("&Log"));
	menu_view->addAction(view_log);
	connect(view_log, &QAction::triggered, this, &MainWindow::viewLogger);
	view_log->setShortcut(Qt::CTRL + Qt::Key_L);
	menu_view->addSeparator();
	auto *view_settings = new QAction(tr("&Settings"));
	menu_view->addAction(view_settings);
	connect(view_settings, &QAction::triggered, this, &MainWindow::viewSettings);

	/* Window menu */
	QMenu *menu_window = this->menuBar()->addMenu(tr("&Window"));
	auto *window_show_workflow = new QAction(tr("Show wor&kflow"));
	menu_window->addAction(window_show_workflow);
	connect(window_show_workflow, &QAction::triggered, this, &MainWindow::showWorkflow);
	window_show_workflow->setShortcut(Qt::CTRL + Qt::Key_K);
	auto *window_hide_workflow = new QAction(tr("&Hide workflow"));
	menu_window->addAction(window_hide_workflow);
	connect(window_hide_workflow, &QAction::triggered, this, &MainWindow::hideWorkflow);
	window_hide_workflow->setShortcut(Qt::CTRL + Qt::Key_H);

#ifdef DEBUG
	/* Debug menu */
	QMenu *menu_debug = this->menuBar()->addMenu("&Debug");
	auto *debug_run = new QAction("&Run action"); //to run arbitrary code with a click
	menu_debug->addAction(debug_run);
	connect(debug_run, &QAction::triggered, this, &MainWindow::z_onDebugRunClick);
#endif //def DEBUG

	/* Help menu */
	QMenuBar *menu_help_main = new QMenuBar(this->menuBar());
	QMenu *menu_help = menu_help_main->addMenu(tr("&Help"));
	auto *help = new QAction(tr("&Help"));
	help->setShortcut(QKeySequence(Qt::Key_F1));
	menu_help->addAction(help);
	connect(help, &QAction::triggered, this, &MainWindow::loadHelp);
	auto *help_about = new QAction(tr("&About"));
	menu_help->addAction(help_about);
	connect(help_about, &QAction::triggered, this, &MainWindow::helpAbout);
	help_about->setShortcut(Qt::Key_F2);
	menu_help->addSeparator();
	auto *help_dev = new QAction(tr("&Developer's help"));
	help_dev->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_F1));
	menu_help->addAction(help_dev);
	connect(help_dev, &QAction::triggered, this, &MainWindow::loadHelpDev);
	auto *help_bugreport = new QAction(tr("File &bug report..."));
	menu_help->addAction(help_bugreport);
	connect(help_bugreport, &QAction::triggered,
	    [=]{ QDesktopServices::openUrl(QUrl("https://models.slf.ch/p/inishell-ng/issues/")); });

	this->menuBar()->setCornerWidget(menu_help_main); //push help menu to the right
}

/**
 * @brief Create the main window's toolbar.
 */
void MainWindow::createToolbar()
{
	/* toolbar */
	toolbar_ = new QToolBar;
	bool toolbar_success;
	Qt::ToolBarArea toolbar_area = static_cast<Qt::ToolBarArea>( //restore last toolbar position
	    getSetting("auto::position::toolbar", "position").toInt(&toolbar_success));
	if (!toolbar_success)
		toolbar_area = Qt::ToolBarArea::TopToolBarArea;
	this->addToolBar(toolbar_area, toolbar_);

	/* user interaction buttons */
	toolbar_->setIconSize(QSize(32, 32));
	QPixmap icon_file;
	icon_file.load(":/icons/ini_file.png");
	toolbar_open_ini_ = toolbar_->addAction(QIcon(icon_file), tr("Open INI file"));
	connect(toolbar_open_ini_, &QAction::triggered, this, [=]{ toolbarClick("open_ini"); });
	toolbar_open_ini_->setEnabled(false); //toolbar entry
	file_open_ini_->setEnabled(false); //menu entry
	toolbar_->addSeparator();
	icon_file.load(":/icons/save.png");
	toolbar_save_ini_ = toolbar_->addAction(QIcon(icon_file), tr("Save INI"));
	toolbar_save_ini_->setEnabled(false); //enable when an INI is open
	connect(toolbar_save_ini_, &QAction::triggered, this, [=]{ toolbarClick("save_ini"); });
	icon_file.load(":/icons/save_as.png");
	toolbar_save_ini_as_ = toolbar_->addAction(QIcon(icon_file), tr("Save INI file as"));
	toolbar_save_ini_as_->setEnabled(false);
	connect(toolbar_save_ini_as_, &QAction::triggered, this, [=]{ toolbarClick("save_ini_as"); });
	icon_file.load(":/icons/preview.png");
	auto *toolbar_preview = toolbar_->addAction(QIcon(icon_file), tr("Preview INI"));
	connect(toolbar_preview, &QAction::triggered, this, [=]{ toolbarClick("preview"); });
	toolbar_->addSeparator();
	icon_file.load(":/icons/clear.png");
	toolbar_clear_gui_ = toolbar_->addAction(QIcon(icon_file), tr("Clear INI settings"));
	connect(toolbar_clear_gui_, &QAction::triggered, this, [=]{ toolbarClick("clear_gui"); });
	toolbar_clear_gui_->setEnabled(false);

	/* info label and autoload checkbox */
	QWidget *spacer = new QWidget;
	spacer->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	QWidget *small_spacer = new QWidget;
	small_spacer->setFixedWidth(25);
	ini_filename_ = new QLabel();
	ini_filename_->setStyleSheet("QLabel {color: " + colors::getQColor("sysinfo").name() + "}");
	ini_filename_->setProperty("mouseclick", "open_ini");
	auto *mouse_events = new MouseEventFilter;
	ini_filename_->installEventFilter(mouse_events);
	ini_filename_->setCursor(QCursor(Qt::PointingHandCursor));
	autoload_box_= new QCheckBox();

	toolbar_->addWidget(spacer);
	toolbar_->addWidget(ini_filename_);
	toolbar_->addWidget(small_spacer);
	autoload_box_ = new QCheckBox();
	connect(autoload_box_, &QCheckBox::stateChanged, this, &MainWindow::onAutoloadCheck);

	autoload_ = toolbar_->addWidget(autoload_box_);
	autoload_->setVisible(false);
	toolbar_->addAction(autoload_);
}

/**
 * @brief Create the main window's status bar.
 */
void MainWindow::createStatusbar()
{ //we use our own because Qt's statusBar is for temporary messages only (they will vanish)
	auto *spacer_widget = new QWidget;
	spacer_widget->setFixedSize(5, 0);
	status_label_ = new QLabel;
	status_label_->setProperty("mouseclick", "open_log");
	auto *mouse_events = new MouseEventFilter;
	status_label_->installEventFilter(mouse_events);
	status_label_->setCursor(QCursor(Qt::PointingHandCursor));
	status_icon_ = new QLabel;
	statusBar()->addWidget(spacer_widget);
	statusBar()->addWidget(status_label_);
	statusBar()->addPermanentWidget(status_icon_);
}

/**
 * @brief Display a text in the main window's status bar.
 * @details Events such as hovering over a menu may clear the status. It is meant
 * for temporary messages.
 * @param[in] message The text to display.
 * @param[in] time Time to display the text for.
 */
void MainWindow::setStatus(const QString &message, const QString &color, const bool &status_light, const int &time)
{ //only temporary messages are possible via statusBar()->showMessage
	status_label_->setText(message);
	status_label_->setStyleSheet("QLabel {color: " + colors::getQColor(color).name() + "}");
	setStatusLight(status_light);
	status_timer_->stop(); //stop pending clearing
	if (time > 0) {
		status_timer_->setInterval(time); //start new interval
		status_timer_->start();
	}
}

/**
 * @brief Set the status light indicating activity on or off.
 * @param[in] on True if it should be active, false for inactive.
 */
void MainWindow::setStatusLight(const bool &on)
{
	const QPixmap icon((on? ":/icons/active.png" : ":/icons/inactive.png"));
	status_icon_->setPixmap(icon.scaled(15, 15));
}

/**
 * @brief Refresh the status label.
 * @details Heavy operations can block the painting which these calls initiate manually.
 */
void MainWindow::refreshStatus()
{
	status_label_->adjustSize();
	status_label_->repaint();
}

/**
 * @brief Clear the main status label.
 */
void MainWindow::clearStatus()
{
	status_label_->setText(QString());
}


//here we could let the user choose an XML file, and add the path to the search directories
//void MainWindow::onMenuOpenXml()
//{
//	QString start_path(getSetting("auto::history::last_menu_xml", "path"));
//	if (start_path.isEmpty())
//		start_path = QDir::currentPath();

//	const QString path = QFileDialog::getOpenFileName(this, tr("Open INI file"), start_path,
//	    "INI files (*.ini);;All files (*)", nullptr, QFileDialog::DontUseNativeDialog | QFileDialog::DontConfirmOverwrite);
//	if (path.isNull()) //cancelled
//		return;
//	openXml(path, QString());

//	const QFileInfo finfo(path);
//	setSetting("auto::history::last_menu_xml", "path", finfo.absoluteDir().path());

//	setWindowTitle(QCoreApplication::applicationName() + tr(" for ") + item->text());
//}

/**
 * @brief Event handler for the "File::Quit" menu: quit the main program.
 */
void MainWindow::quitProgram()
{
	QApplication::quit();
}

/**
 * @brief Event handler for the "GUI::resetGui" menu: reset thr GUI completely.
 * @details This is the only action that resets the GUI to the starting point.
 */
void MainWindow::resetGui()
{
	toolbarClick("clear_gui");
	control_panel_->clearGuiElements();
	control_panel_->displayInfo();
	help_loaded_ = false;
}

/**
 * @brief Open the preview window.
 */
void MainWindow::viewPreview()
{
	preview_->addIniTab();
	preview_->show();
	preview_->raise();
}

/**
 * @brief Event handler for the "View::Log" menu: open the Logger window.
 */
void MainWindow::viewLogger()
{
	logger_.show(); //the logger is always kept in scope
	logger_.raise(); //bring to front
}

/**
 * @brief Event listener for when the program is closed.
 * @details This function checks whether there are unsaved changes before exiting the program,
 * and if yes, offers the usual options.
 * @param[in] event The received close event.
 */
void MainWindow::closeEvent(QCloseEvent *event)
{
	const bool perform_close = closeIni();
	if (perform_close)
		event->accept();
	else
		event->ignore();
}

/**
 * @brief Event listener for key events.
 * @param event The key press event that is received.
 */
void MainWindow::keyPressEvent(QKeyEvent *event)
{
	if (event->key() == Qt::Key_Escape) { //ESC to get out of the help file
		if (help_loaded_)
			resetGui();
	}
}

/**
 * @brief Event handler for the "View::Settings" menu: open the settings dialog.
 */
void MainWindow::viewSettings()
{
	static auto *settings_dialog = new Settings(this); //allow only once, load when asked
	settings_dialog->show();
	settings_dialog->raise();
}

/**
 * @brief Show the workflow panel in case it is collapsed.
 */
void MainWindow::showWorkflow()
{
	int main_width = this->width(); //the workflow has a maximum size anyway so we just set someting big
	QList<int> splitter_sizes = {main_width / 2 , main_width / 2};
	control_panel_->setSplitterSizes(splitter_sizes);
}

/**
 * @brief Hide the workflow panel.
 */
void MainWindow::hideWorkflow()
{
	const int main_width = this->width();
	control_panel_->setSplitterSizes(QList<int>() << 0 << main_width);
}

/**
 * @brief Load the user help XMl from the resources onto the GUI.
 */
void MainWindow::loadHelp()
{
	clearGui();
	openXml(":doc/help.xml", "Help");
	help_loaded_ = true;
}

/**
 * @brief Load the developer help XMl from the resources onto the GUI.
 */
void MainWindow::loadHelpDev()
{
	clearGui();
	openXml(":doc/help_dev.xml", "Help");
	help_loaded_ = true;
}

/**
 * @brief Display the 'About' window.
 */
void MainWindow::helpAbout()
{
	static auto *about = new AboutWindow(this);
	about->show();
	about->raise();
}

/**
 * @brief Delegate the toolbar button clicks to the appropriate calls.
 * @param[in] function Tag name for the function to perform.
 */
void MainWindow::toolbarClick(const QString &function)
{
	if (function == "save_ini")
		saveIni();
	else if (function =="save_ini_as")
		saveIniAs();
	else if (function == "open_ini")
		openIni();
	else if (function == "clear_gui")
		clearGui();
	else if (function == "preview")
		viewPreview();
}

/**
 * @brief Event listener for when the 'autoload' checkbox is clicked.
 * @details This function sets settings for the INI files to be automatically loaded for certain XMLs.
 * They are saved when the program quits.
 * @param[in] state The checkbox state (checked/unchecked).
 */
void MainWindow::onAutoloadCheck(const int &state)
{
	//Run through <ini> tags in appropriate settings file section and look for the current
	//application. If it is found, when the checkbox is checked it will be set to the current
	//INI file. If the box is unchecked the entry is deleted:
	QDomNode autoload_node = xml_settings_.firstChildElement().firstChildElement("user").firstChildElement("autoload");
	for (auto ini = autoload_node.firstChildElement("ini"); !ini.isNull(); ini = ini.nextSiblingElement("ini")) {
		if (ini.attribute("application").toLower() == current_application_.toLower()) {
			if (state == Qt::Checked) {
				ini.firstChild().setNodeValue(ini_filename_->text());
				return;
			} else {
				ini.parentNode().removeChild(ini);
				return;
			}
		}
	}

	//not found - create new settings node if user is enabling:
	if (state == Qt::Checked) {
		QDomNode ini = autoload_node.appendChild(xml_settings_.createElement("ini"));
		ini.toElement().setAttribute("application", current_application_);
		ini.appendChild(xml_settings_.createTextNode(ini_filename_->text()));
	}
}

#ifdef DEBUG
/**
 * @brief This is a dummy menu entry to execute arbitrary code from.
 */
void MainWindow::z_onDebugRunClick()
{
	QColor col(Qt::darkRed);
	std::cout << col.name().toStdString() << std::endl;
	setStatus("Debug menu clicked", "red", false, 5000);
}
#endif //def DEBUG
