/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * INIshell - A dynamic graphical interface to generate INI files
 * INIshell v2: Michael Reisecker, 2019
 * Inspired by INIshell v1: Mathias Bavay, Thomas Egger & Daniela Korhammer, 2011
 *
 * This is the main program starting the event loop.
 * 2019-10
 */

#include "colors.h"
#include "Error.h"
#include "XMLReader.h"
#include "src/gui/MainWindow.h"
#include "src/gui/Settings.h"

#include <QApplication>
#include <QCommandLineParser>
#include <QFile>
#include <QTranslator>

#include <iostream>

/**
 * @brief Set meta data for the application.
 * @details This is used in different places INIshell writes and installs to depending on the OS.
 */
void setAppMetadata()
{
	QApplication::setApplicationName("INIshell");
	QApplication::setOrganizationName("SLF");
	QApplication::setOrganizationDomain("slf.ch");
	QApplication::setApplicationVersion(APP_VERSION_STR);
	QApplication::setWindowIcon(QIcon(":/icons/inishell.ico"));
}

/**
 * @brief Prepare the command line parser options.
 * @param[in] parser Instance of the command line parser.
 * @param[in] cmd_args Command line args given to the programs.
 */
void prepareCommandline(QCommandLineParser &parser, command_line_args &cmd_args)
{
	QCommandLineOption opt_i({"i", "inifile"}, "INI file to import on startup", "inifile");
	QCommandLineOption opt_s({"s", "settingsfile"}, "INIshell settings file", "settingsfile");
	QCommandLineOption opt_o({"o", "outinifile"}, "INI file to write out, surpassing the GUI", "outinifile");
	parser.addOption(opt_i);
	parser.addOption(opt_s);
	parser.addOption(opt_o);
	parser.addHelpOption();
	parser.addVersionOption();
	parser.process(QCoreApplication::arguments());
	cmd_args.startup_ini_file = parser.value("inifile");
	cmd_args.settings_file = parser.value("settingsfile");
	cmd_args.out_ini_file = parser.value("outinifile");
}

/**
 * @brief Load settings from INIshell's XML settings file.
 * @details This has nothing to do with the XMLs that are parsed to build the GUI,
 * stored here are INIshell's own static GUI settings (like the language etc.).
 * @param cmd_args
 * @return
 */
QDomDocument loadSettings(const command_line_args &cmd_args)
{
	/* find the settings file to use */
	const QString filename("inishell_settings.xml");
	QString settings_file(":/" + filename); //default: embedded file
	if (!cmd_args.settings_file.isEmpty()) //file given in command line
		settings_file = cmd_args.settings_file;
	else if (QFile::exists("./" + filename))
		settings_file = "./" + filename;
	XMLReader xml_settings;
	QString xml_error; //TODO: display this
	const bool read_success = xml_settings.read(settings_file, xml_error, true);
	if (!read_success)
		Error(QApplication::tr("Could not read settings file."), QApplication::tr("Unable to open file ") + settings_file,
		    "", error::warning);
	return xml_settings.getXml();
}

/**
 * @brief Set global stylesheets for panels/widgets.
 * @details Global styling is done here, including the styles of properties that may or may not
 * be set, such as how a mandatory field that is not filled in should look. Further styling is
 * done locally.
 * @param[in] app The main app instance.
 */
void setAppStylesheet(QApplication &app)
{
	/*
	 * Set the global stylesheet:
	 *  (1) All widgets that have their on-the-fly property "mandatory" set will look like this
	 *  (2) All panels that are currently set to the default value will look like this
	 *  (3) The main scroll areas and all their (grand-) children will have this color
	 *  (4), (5) Except for their scroll bars
	 *  (6) Grouping elements will have this default color
	 */
	app.setStyleSheet(" \
	    * [mandatory=\"true\"] {background-color: " + colors::getQColor("mandatory").name() + "; color: " +
	    colors::getQColor("mandatory_text").name() + "} \
	    * [shows_default=\"true\"] {background-color: " + colors::getQColor("default_values").name() + "} \
	    * [faulty=\"true\"] {background-color: " + colors::getQColor("faulty_values").name() + "} \
	    * [valid=\"true\"] {background-color: " + colors::getQColor("valid_values").name() + "} \
	    QScrollArea {background-color: white} \
	    QScrollBar:vertical {background: #f2f2f2} \
	    QScrollBar:horizontal {background: #f2f2f2; height: 15px;} \
	    Group {background-color: white} \
	");
}
//	    * QComboBox {background-color: white}

/**
 * @brief Entry point of the main program.
 * @details This function starts the main event loop.
 * @param[in] argc Command line arguments count.
 * @param[in] argv Command line arguments.
 * @return Exit code.
 */
int main(int argc, char *argv[])
{
	QApplication app(argc, argv);

	/* parse the command line */
	QCommandLineParser parser;
	command_line_args cmd_args;
	prepareCommandline(parser, cmd_args);

	/* load and apply settings for the static part of the GUI */
	QDomDocument settings = loadSettings(cmd_args); //TODO: log errors from here

	const QString in_inifile = cmd_args.startup_ini_file;
	INIParser cmd_ini;
	if (!in_inifile.isEmpty())
		cmd_ini.setFile(in_inifile);
	const QString out_inifile = cmd_args.out_ini_file;
	if (out_inifile.isEmpty()) { //GUI mode
		const QString language = getSetting(settings, "appearance", "language");
		QTranslator translator; //can't go out of scope
		if (!language.isEmpty() && language != "en") { //texts that are not found will remain in English
			const QString language_file(":/languages/inishell_" + language);
			if (translator.load(language_file))
				QApplication::installTranslator(&translator);
			else
				Error("Language file not found", "File \"" + language_file +
				    "\" is not a valid language file.");
		} //endif language

		setAppMetadata();
		setAppStylesheet(app);
		MainWindow main_window(settings);
		main_window.show(); //start INIshell's GUI
		return QApplication::exec();
	} else { //stay in the terminal
		if (!in_inifile.isEmpty()) {
			QFile ini_output(out_inifile);
			if (ini_output.open(QIODevice::WriteOnly)) {
				QTextStream iniss(&ini_output);
				cmd_ini.outputIni(iniss);
			}
		} else {
			std::cerr << QCoreApplication::tr(
			    "[E] Can only output to file via -o if an input file is set via -i.").toStdString() << std::endl;
		}

		return 0;
	} //endif out_ini_file
}
