/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "XMLReader.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

#include <QCoreApplication> //for translations

#ifdef DEBUG
	#include <QDebug>
	#include <iostream>
#endif

QDomNode prependParent(QDomNode &child)
{
	QByteArray parent_array("<dummy_parent></dummy_parent>");
	QDomDocument parent_doc;
	parent_doc.setContent(parent_array, false);
	parent_doc.firstChild().appendChild(child.cloneNode(true));
	return parent_doc.firstChild();
}

XMLReader::XMLReader(const QString &filename, QString &xml_error)
{
	this->read(filename, xml_error);
}

bool XMLReader::read(const QString &filename, QString &xml_error, const bool &no_references)
{
	QFile infile(filename);
	QFileInfo finfo(infile);
	master_xml_path_ = finfo.path();
	if (!infile.open(QIODevice::ReadOnly | QIODevice::Text))
		return false;
	const bool success = this->read(infile, xml_error, no_references);
	infile.close();
	return success;
}

bool XMLReader::read(QFile &file, QString &xml_error, const bool &no_references)
{
	QString error_msg;
	int error_line, error_column;
	if (!xml_.setContent(&file, false, &error_msg, &error_line, &error_column)) {
		xml_error = QString(QCoreApplication::tr( //TODO: macro
		    "%1 (line %2, column %3)")).arg(error_msg).arg(error_line).arg(error_column);
		return false;
	}
	parseIncludes();
	if (!no_references) //e. g. static GUI settings file
		parseReferences();
	xml_error = QString();
	return true;
}

void XMLReader::parseReferences()
{ //TODO: done quick & dirty
	QDomNodeList par_groups = getXml().elementsByTagName("parametergroup");
	QDomNodeList to_substitute = getXml().elementsByTagName("reference");
	for (int ii = 0; ii < to_substitute.size(); ++ii) {
		const QString sub_name(to_substitute.at(ii).toElement().attribute("name"));
		for (int jj = 0; jj < par_groups.size(); ++jj) { //check nodes we can refer to
			if (par_groups.at(jj).toElement().attribute("name").toLower() == sub_name.toLower()) {
				for (QDomNode child = par_groups.at(jj).firstChild(); !child.isNull(); child = child.nextSibling())
					to_substitute.at(ii).parentNode().appendChild(child.cloneNode()); //TODO: shallow copy
				to_substitute.at(ii).appendChild(par_groups.at(jj)); //shallow copy
			}
		}
	} //endfor to_substitute
}

void XMLReader::parseIncludes()
{
	while (true) {
		QDomNodeList includes = xml_.elementsByTagName("include");
		if (includes.isEmpty())
			break;
		for (int ii = 0; ii < includes.count(); ++ii) {
			const QString include_file_name(includes.at(ii).toElement().attribute("file"));
			QFile include_file(master_xml_path_ + "/" + include_file_name);
			if (include_file.open(QIODevice::ReadOnly)) {
				QDomDocument inc;
				inc.setContent(include_file.readAll());
				QDomDocumentFragment new_fragment = inc.createDocumentFragment();
				for (QDomNode nd = inc.firstChild().firstChild(); !nd.isNull(); nd = nd.nextSibling())
					new_fragment.appendChild(nd.cloneNode(true));
				const QDomNode success = includes.at(ii).parentNode().replaceChild(new_fragment, includes.at(ii));
#ifdef DEBUG
				if (success.isNull())
					qDebug() << "Replacing a node failed for inclusion system in file " << include_file_name << endl;
#endif //def DEBUG
			} else {
				topLog(QString(QCoreApplication::tr("Unable to open XML include file \"%1\" for reading").arg(include_file_name)),
				    "xmlerror");
			}
		}
	} //end while
}

QDomDocument XMLReader::getXml() const
{
	return xml_;
}

