////////////////////////////////////////
///          SELECTOR panel          ///
////////////////////////////////////////

#include "Selector.h"
#include "Label.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QPushButton>

/**
 * @class Selector
 * @brief Default constructor for a Selector.
 * @details A selector panel allows to select from a list of text pieces (e. g. meteo parameters), either from
 * a fixed dropdown list or with possible free text input. Its single child panel must be declared as "template"
 * in its attributes. With the click of a button the child element is duplicated, inheriting the text piece.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Selector.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Selector::Selector(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label, dropdown menu and buttons */
	auto *key_label = new Label(section_, "_selector_label_" + key_, options, no_spacers, key_);
	dropdown_ = new QComboBox;
	setPrimaryWidget(dropdown_, false); //set ID later
	dropdown_->setMinimumWidth(Cst::tiny); //no tiny elements
	dropdown_->setEditable(true); //free text with autocomplete
	dropdown_->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed); //size of biggest item
	auto *plus_button = new QPushButton("+"); //TODO: icons
	auto *minus_button = new QPushButton("-");
	plus_button->setFixedSize(dropdown_->height(), dropdown_->height()); //buttons have same height as dropdown
	minus_button->setFixedSize(plus_button->size());
	connect(plus_button, &QPushButton::clicked, this, &Selector::guiAddPanel);
	connect(minus_button, &QPushButton::clicked, this, &Selector::removePanel);

	/* layout of the basic elements */
	auto *selector_layout = new QHBoxLayout;
	setLayoutMargins(selector_layout);
	selector_layout->addWidget(key_label);
	selector_layout->addWidget(dropdown_, 0, Qt::AlignLeft);
	selector_layout->addWidget(plus_button, 0, Qt::AlignLeft);
	selector_layout->addWidget(minus_button, 0, Qt::AlignLeft);
	if (!no_spacers)
		selector_layout->addSpacerItem(buildSpacer()); //keep buttons from wandering to the right

	/* layout for the basic elements plus children */
	container_ = new Group(section_, "_selector_" + key_, true);
	container_->setVisible(false); //only visible with items
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(selector_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //construct children
}

/**
 * @brief Parse options for a Selector from XML.
 * @param[in] options XML node holding the Selector.
 * @return True if all options were set successfully.
 */
bool Selector::setOptions(const QDomNode &options)
{
	//TODO: optionally no free text
	for (QDomElement par = options.firstChildElement("parameter"); !par.isNull(); par = par.nextSiblingElement("parameter")) {
		if (par.attribute("template") == "true") {
			templ_ = par; //save the node describing the child (shallow copy)
			this->setObjectName(getQtKey(section_ + Cst::sep + par.attribute("key")));
			break; //TODO: check for bad syntax (multiple templates)
		}
	}
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option"))
		dropdown_->addItem(op.attribute("value")); //fill list of texts the selector shows as default
	return true;
}

/**
 * @brief Event listener for the plus button: add a child panel to the selector.
 * @details This function replicates the child panel from XML and passes the selected text.
 */
void Selector::guiAddPanel()
{
	const QString param_text(dropdown_->currentText());
	if (param_text.isEmpty()) {
		topStatus(tr("Empty field in dropdown text"));
		return;
	}
	if (container_map_.count(param_text) != 0) { //only one panel per piece of text
		topStatus(tr("Item already exists"));
		return;
	}
	topStatus(""); //above messages could be confusing if still displayed from recent click
	addPanel(param_text);
}

void Selector::iniAddPanel(const QString &param_text)
{
	//TODO: log ini errors
	addPanel(param_text);
}

void Selector::addPanel(const QString &param_text)
{
	//we clone the child node (deep copy for string replacement) and put it in a dummy parent:
	QDomNode node(prependParent(templ_)); //nest for recursion
	QDomElement element(node.toElement());
	substituteKeys(element, "%", param_text); //recursive substitution for all children
	 //draw it on next call (don't use as template again):
	node.firstChildElement("parameter").setAttribute("template", "false");

	/* construct all children and grandchildren */
	Group *new_group = new Group(section_, "_selector_panel_" + key_);
	recursiveBuild(node, new_group, section_);
	container_->addWidget(new_group);
	container_->setVisible(true);

	const auto panel_pair = std::make_pair(param_text, new_group);
	container_map_.insert(panel_pair); //keep an index of text to panel number
}

/**
 * @brief Event listener for the minus button: remove a panel for the selected text piece.
 * @details If some text is present in the dropdown menu, the corresponding child is deleted.
 */
void Selector::removePanel()
{
	const QString drop_text(dropdown_->currentText());
	auto it = container_map_.find(drop_text); //look up if item exists in map
	if (it != container_map_.end()) {
		topStatus(""); //no "does not exist" error message from earlier
		it->second->erase(); //delete the group's children
		delete it->second; //delete the group itself
		container_map_.erase(it);
		if (container_map_.empty()) //no more children - save a couple of blank pixels
			container_->setVisible(false);
	} else {
		topStatus(tr("Item does not exist"));
	}
}

void Selector::onPropertySet()
{
	const QString panel_to_add = this->property("ini_value").toString();
//	std::cout << "Selector::onPropertySet: " << panel_to_add.toStdString() << std::endl;
	iniAddPanel(panel_to_add);
}
