////////////////////////////////////////
///          LABEL element           ///
////////////////////////////////////////

#include "Label.h"
#include "src/main/colors.h"

#include <QVBoxLayout>

/**
 * @class Label
 * @brief Default constructor for a Label.
 * @details A label displays styled text, for example for INI keys.
 * @param[in] section INI section the described value belongs to. Could be used for style targeting
 * but is ignored otherwise.
 * @param[in] key INI key corresponding to the value that is being described. If no label is set then
 * this text will be displayed.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] in_label Label to display instead of the key.
 * @param[in] parent The parent widget.
 */
Label::Label(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    const QString &in_label, QWidget *parent) : Atomic(section, key, parent)
{
	/* label styling */
	QString label(options.toElement().attribute("label"));
	if (label.isNull()) //probably not read from XML
		label = in_label.isNull()? key_ : in_label;

	label_ = new QLabel(label);
	QPalette label_palette(this->palette());
	label_palette.setColor(QPalette::WindowText, colors::getQColor("key"));
	label_->setPalette(label_palette);
	label_->setFont(setFontOptions(label_->font(), options.toElement()));

	/* main layout */
	auto *layout = new QVBoxLayout;
	layout->addWidget(label_);
	setLayoutMargins(layout);
	this->setLayout(layout);
	if (!no_spacers)
		this->setMinimumWidth(getColumnWidth(label, Cst::label_width)); //set to minimal required room
	//set a fixed with so that window resizing does not introduce unneccessary margins:
	this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

/**
 * @brief Check if a label actually has text.
 * @return True if the label displays some text.
 */
bool Label::isEmpty()
{
	return label_->text().isEmpty();
}

/**
 * @brief Retrieve the minimal width required for the label.
 * @details A fixed minimal width is set for labels so that they act as the first "column" of the interface
 * making it more visually appealing. If a label needs more space than that it is extended.
 * @param[in] text Text that will be displayed.
 * @param[in] min_width The minimum "column" width for the (INI key) labels.
 * @return The size to set the label to.
 */
int Label::getColumnWidth(const QString &text, const int& min_width)
{
	const QFontMetrics font_metrics(this->font());
	const int text_width = font_metrics.boundingRect(text).width();
	if (text_width > min_width)
		return text_width + Cst::label_padding;
	else
		return min_width + Cst::label_padding;
}
