////////////////////////////////////////
///         FILE/PATH panel          ///
////////////////////////////////////////

#include "FilePath.h"
#include "Label.h"
#include "src/main/colors.h"

#include <QFileDialog>
#include <QHBoxLayout>
#include <QPushButton>
/**
 * @class FilePath
 * @brief Default constructor for a file/path picker.
 * @details The file/path picker displays a dialog to select either a file or a folder.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this file/path picker.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
FilePath::FilePath(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label, text field, "open" button and info label */
	auto *key_label = new Label(section_, key_, options, no_spacers, key_);
	path_text_ = new QLineEdit; //textfield with one line
	setPrimaryWidget(path_text_);
	auto *open_button = new QPushButton("...");
	open_button->setFixedSize(Cst::medium_button_width, Cst::std_button_height);
	connect(open_button, &QPushButton::clicked, this, &FilePath::openFile);
	info_text_ = new QLabel();
	info_text_->setAlignment(Qt::AlignCenter);
	QPalette label_palette(info_text_->palette()); //custom color
	label_palette.setColor(QPalette::WindowText, colors::getQColor("iniwarning"));
	info_text_->setPalette(label_palette);
	info_text_->setVisible(false); //TODO: flickering

	/* layout of the basic elements */
	auto *filepath_layout = new QHBoxLayout;
	setLayoutMargins(filepath_layout);
	filepath_layout->addWidget(key_label, 0, Qt::AlignLeft);
	filepath_layout->addWidget(path_text_);
	filepath_layout->addWidget(open_button, 0, Qt::AlignLeft);
	addHelp(filepath_layout, options);

	/* main layout with interactive widgets and info label */
	auto *main_layout = new QVBoxLayout;
	main_layout->addLayout(filepath_layout);
	main_layout->addWidget(info_text_, 0, Qt::AlignLeft);
	setLayoutMargins(main_layout);
	this->setLayout(main_layout);

	setOptions(options); //file or path
}

/**
 * @brief Parse options for a file/patch picker from XML.
 * @param[in] options XML node holding the file/path picker.
 * @return True if all options were set successfully.
 */
bool FilePath::setOptions(const QDomNode &options)
{
	if (options.toElement().attribute("type") == "path")
		path_only_ = true;
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		const QString ext = op.attribute("extension"); //selectable file extensions can be set
		extensions_ += ext + (ext.isEmpty()? "" : ";;");
	}
	if (extensions_.isEmpty())
		extensions_= "All Files (*)";
	else
		extensions_.chop(2); //remove trailing ;;
	connect(path_text_, &QLineEdit::textEdited, this, &FilePath::checkValue);
	setDefaultPanelStyles(path_text_->text());
	return true;
}
void FilePath::onPropertySet()
{
	const QString filename = this->property("ini_value").toString();
	path_text_->setText(filename);
	checkValue(filename);
}

void FilePath::checkValue(const QString &filename)
{
	setDefaultPanelStyles(filename);

	//setUpdatesEnabled(false);
	if (filename.isEmpty()) {
		info_text_->setText(tr("[No file set]"));
		info_text_->setVisible(true); //TODO: flickering
		//setUpdatesEnabled(true);
		//do not return, we also save the empty field
	}
	path_text_->setText(filename);
	if (!QFile::exists(filename) && !filename.isEmpty()) {
		info_text_->setText(path_only_? tr("[Folder does not exist]") : tr("[File does not exist]"));
		info_text_->setVisible(true);
	} else {
		info_text_->setText(QString());
	}
	setIniValue(filename); //the label is just input -> file does not actually have to exist
	//info_text_->setVisible(false);
	//setUpdatesEnabled(true);
}

/**
 * @brief Event listener for the open button.
 * @details Open a file or path by displaying a dialog window.
 */
void FilePath::openFile()
{
	path_text_->setProperty("shows_default", "true");

	QString path;
	if (path_only_) { //TODO: remember last location
		path = QFileDialog::getExistingDirectory(this, tr("Open Folder"), "./",
		    QFileDialog::DontUseNativeDialog | QFileDialog::ShowDirsOnly);
	} else {
		path = QFileDialog::getOpenFileName(this, tr("Open File"), "./",
		    extensions_, nullptr, QFileDialog::DontUseNativeDialog);
	}
	this->setProperty("ini_value", path);
}
