////////////////////////////////////////
///          DROPDOWN panel          ///
////////////////////////////////////////

#include "Dropdown.h"
#include "src/main/constants.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

#include <QColor>
#include <QFont>
#include <QStringList>

#include <algorithm>
#include <numeric> //for iota

/**
 * @class Dropdown
 * @brief Default constructor for a Dropdown panel.
 * @details A Dropdown panel shows a dropdown menu where additional options pop up depending on the
 * selected menu item.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Choice panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Dropdown::Dropdown(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and combo box */
	dropdown_ = new QComboBox;
	setPrimaryWidget(dropdown_);
	dropdown_->setStyleSheet("");
//	dropdown_->setStyleSheet("[shows_default=\"true\"] {background: " + colors::getQColor("default_values").name() + "; color: black}");
//	dropdown_->setStyleSheet("QComboBox {color: black}");
//	dropdown_->setStyleSheet("QAbstractItemView {background-color: " + getMainWindow()->palette().color(QPalette::Window).name() + "}");


	dropdown_->setMinimumWidth(Cst::tiny); //no tiny elements
	connect(dropdown_, QOverload<int>::of(&QComboBox::currentIndexChanged), this, &Dropdown::itemChanged);

	/* layout of the basic elements */
	auto *dropdown_layout = new QHBoxLayout;
	setLayoutMargins(dropdown_layout);
	if (!key.isEmpty()) {
		auto *key_label = new Label(section_, key_, options, no_spacers, key_);
		dropdown_layout->addWidget(key_label, 0, Qt::AlignLeft);
	}
	dropdown_layout->addWidget(dropdown_, 0, Qt::AlignLeft);
	//add a big enough spacer to the right to keep the buttons to the left (unless it's a horizontal layout):
	if (!no_spacers)
		dropdown_layout->addSpacerItem(buildSpacer());
	addHelp(dropdown_layout, options);

	/* layout of the basic elements plus children */
	container_ = new Group(section, "_dropdown_" + key_, true, false, false, true); //tight layout
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(dropdown_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //construct child panels
	emit itemChanged(0); //select first option
}

/**
 * @brief Parse options for a Dropdown panel from XML.
 * @param[in] options XML node holding the Dropdown panel.
 * @return True if all options were set successfully.
 */
bool Dropdown::setOptions(const QDomNode &options)
{
	//To set item fonts we need to do it for the AbstractListModel (the popup),
	//and dynamically change the QComboBox font when selecting the item.
	QStringList item_strings; //to size it
	if (options.toElement().attribute("optional") != "false") {
		QFont dummy_font = dropdown_->font();
		dummy_font.setItalic(true);
		dropdown_->addItem("<select>");
		dropdown_->setItemData(0, dummy_font, Qt::FontRole);
		Group *dummy_group = new Group(section_, "_dummy_group_" + key_, false, false, false, true); //tight layout
		container_->addWidget(dummy_group);
		item_strings.push_back("<select>");
	}
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		/* add childrens' names to dropdown and build them */
		dropdown_->addItem(op.attribute("value"));
		item_strings.push_back(op.attribute("value"));
		QFont item_font = setFontOptions(dropdown_->font(), op);
		dropdown_->setItemData(dropdown_->count() - 1, item_font, Qt::FontRole);
		//dropdown_->setItemData(dropdown_->count() - 1, colors::getQColor(op.attribute("color")), Qt::ForegroundRole);

		auto *item_group = new Group(section_, "_item_dropdown_" + key_); //group all elements of this option together
		recursiveBuild(op, item_group, section_);
		container_->addWidget(item_group);
	}
	dropdown_->setMinimumWidth(getElementTextWidth(item_strings, Cst::tiny, Cst::maximum_dropdown_width) + Cst::dropdown_safety_padding);
	return true;
}

/**
 * @brief Event listener for when a dropdown menu item is selected.
 * @details This function shows/hides the children corresponding to the selected dropdown item.
 * @param[in] index Index/number of the selected item.
 */
void Dropdown::itemChanged(int index)
{
	QLayout *group_layout = container_->getLayout();
	for (int ii = 0; ii < group_layout->count(); ++ii) {
		//find the group to display and hide all others:
		auto *group_item = qobject_cast<Group *>(group_layout->itemAt(ii)->widget());
		group_item->setVisible(index == ii);
		if (ii == index) //hide the container if the displayed group is empty
			container_->setVisible(!group_item->isEmpty());
	}

	//set the QComboBox font to the selected item's font:
	dropdown_->setFont(qvariant_cast<QFont>(dropdown_->itemData(index, Qt::FontRole)));
//	QColor color = qvariant_cast<QColor>(dropdown_->itemData(index, Qt::ForegroundRole));
//	dropdown_->setStyleSheet("QComboBox {color: " + color.name() + "}"); //TODO: can't see how to do this

	const QString dropdown_text(dropdown_->currentText());
	setDefaultPanelStyles(dropdown_text);
	setIniValue(dropdown_text == "<select>"? "" : dropdown_text);
}

void Dropdown::onPropertySet()
{
	const QString text_to_set = this->property("ini_value").toString();
	if (text_to_set.isEmpty()) { //default="" in XML --> do nothing if optional
		if (this->property("is_mandatory") == "false")
			return;
	}
	for (int ii = 0; ii < dropdown_->count(); ++ii) {
		if (dropdown_->itemText(ii).toLower() == text_to_set.toLower()) {
			dropdown_->setCurrentIndex(ii);
			return;
		}
	}
	topLog(tr("Alternative value \"%1\" could not be set from INI file for key \"%2\": no such option specified in XML file").arg(
	    text_to_set).arg(key_), "iniwarning");
}
