/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Checkbox.h"
#include "Label.h"

#include "src/main/colors.h"

#include <QHBoxLayout>

/**
 * @class Checkbox
 * @brief Default constructor for a Checkbox panel.
 * @details A Checkbox is a single on/off switch.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Checklist.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Checkbox::Checkbox(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and checkbox */
	auto *key_label = new Label(section_, "_checkbox_key_" + key_, options, no_spacers, key_);
	QString caption (options.toElement().attribute("caption"));
	if (caption.isEmpty())
		caption = key;
	checkbox_ = new QCheckBox(caption);
	setPrimaryWidget(checkbox_);
	setFontOptions(primary_widget_, options);

	auto *checkbox_layout = new QHBoxLayout;
	setLayoutMargins(checkbox_layout);
	if (!key_label->isEmpty())
		checkbox_layout->addWidget(key_label, 0, Qt::AlignLeft);
	checkbox_layout->addWidget(checkbox_, 0);

	/* spacer and help text */
	if (!no_spacers)
		checkbox_layout->addSpacerItem(buildSpacer()); //keep widgets to the left
	addHelp(checkbox_layout, options);
	this->setLayout(checkbox_layout);

	setOptions(options);
}

/**
 * @brief Parse options for a Checkbox from XML.
 * @param[in] options XML node holding the Checkbox.
 * @return True if all options were set successfully.
 */
bool Checkbox::setOptions(const QDomNode &/*options*/)
{
	connect(checkbox_, &QCheckBox::stateChanged, this, &Checkbox::checkValue);
	return true;
}

/**
 * @brief Event listener for changed INI values.
 * @details The "ini_value" property is set when parsing default values and potentially again
 * when setting INI keys while parsing a file.
 */
void Checkbox::onPropertySet()
{
	const QString value = this->property("ini_value").toString();
	if (value.toLower() == "true" || value == "1")
		checkbox_->setCheckState(Qt::Checked);
	else
		checkbox_->setCheckState(Qt::Unchecked);

	/* keep desired output format (1/0 or true/false or TRUE/FALSE) */
	if (value == "1" || value == "0")
		numeric_ini_value_ = true;
	else //so that the INI can overpower the XML format
		numeric_ini_value_ = false;
	if (value.isLower())
		lowercase_ini_value_ = true;
	else
		lowercase_ini_value_ = false;

	checkValue(checkbox_->checkState());
}

/**
 * @brief Handle changes to the INI value through this panel.
 * @details This function is called as an event listener by the primary widget when the user
 * interacts with it and also when the property watcher reports changes, i. e. at all times
 * the INI value is requested to change. It is the final layer before handling the value to
 * the INIParser and can therefore perform checks and formatting.
 * @param[in] state The checkbox state (checked, unchecked, or tristate)
 */
void Checkbox::checkValue(const int &state)
{
	QString user_state; //user formatted INI value from checkbox state
	if (state == Qt::Checked)
		user_state = numeric_ini_value_? "1" : "TRUE";
	else
		user_state = numeric_ini_value_? "0" : "FALSE";
	if (lowercase_ini_value_)
		user_state = user_state.toLower();

	setDefaultPanelStyles(user_state);
	setIniValue(user_state);
}
