/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "gui_elements.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"
#include "src/gui/Logger.h"

#include <QApplication>

/**
 * @brief Object factory for the panels.
 * @details Construct an object from a string name (often read from an XML).
 * @param[in] identifier Name of the object.
 * @param[in] section INI section the controlled values belong to.
 * @param[in] key INI key of the controlled value.
 * @param[in] options The current XML node with all options and children.
 * @param[in] no_spacers If available, set a tight layout for the object.
 * @param[in] parent The parent widget.
 * @return An object of the panel family to manipulate values.
 */
QWidget * elementFactory(const QString &identifier, const QString &section, const QString &key,
    const QDomNode &options, const bool& no_spacers, QWidget *parent)
{
	QWidget *element;
	if (options.toElement().attribute("replicate") == "true") {
		element = new Replicator(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "alternative") {
		element = new Dropdown(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "checklist") {
		element = new Checklist(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "choice") {
		element = new Choice(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "file" || identifier.toLower() == "path") {
		element = new FilePath(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "grid") {
		element = new GridPanel(section, key, options, parent);
	//TODO: Helptext
	} else if (identifier.toLower() == "horizontal") {
		element = new HorizontalPanel(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "label") {
		element = new Label(section, key, options, no_spacers, QString(), parent); //no forced caption
	} else if (identifier.toLower() == "number") {
		element = new Number(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "selector") {
		element = new Selector(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "text") {
		element = new Textfield(section, key, options, no_spacers, parent);
	} else {
		topLog(QApplication::tr("Unknown parameter object in XML file: \"") + identifier +
		    "\"", colors::getQColor("error"));
		topStatus(QApplication::tr("Unknown XML parameter type"));
		element = nullptr; //will throw a Qt warning
	}

	if (element != nullptr) {
		const QString &default_value = options.toElement().attribute("default");
		if (!default_value.isEmpty()) {
			element->setProperty("default_value", default_value);
			element->setProperty("ini_value", default_value);
		}
		const bool &is_mandatory = (options.toElement().attribute("optional") == "false");
		if (is_mandatory) {
			element->setProperty("is_mandatory", "true");
		}
		Atomic *atomic_element = static_cast<Atomic *>(element);
		atomic_element->setDefaultPanelStyles(default_value);
	}
	return element;
}
