////////////////////////////////////////
///         REPLICATOR panel         ///
////////////////////////////////////////

#include "Replicator.h"
#include "Label.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QPushButton>

/**
 * @class Replicator
 * @brief Default constructor for a Replicator.
 * @details A Replicator holds a widget which it can replicate with the click of a button.
 * It does not have a separate identifier but rather it is activated in any given panel with the "replicate"
 * attribute. The number of the created panel is propagated to all children via "#".
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Replicator.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Replicator::Replicator(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	container_ = new Group(section_, "_replicator_" + key, true);

	/* label, dropdown menu and buttons */
	auto *key_label = new Label(section_, "_replicator_label_" + key_, options, no_spacers, key_);
	setPrimaryWidget(key_label, false);

	auto *plus_button = new QPushButton("+");
	auto *minus_button = new QPushButton("-");
	plus_button->setFixedSize(Cst::std_button_width, Cst::std_button_height);
	minus_button->setFixedSize(plus_button->size());
	connect(plus_button, &QPushButton::clicked, this, &Replicator::replicate); //replicate child
	connect(minus_button, &QPushButton::clicked, this, &Replicator::deleteLast); //delete last child

	/* layout of the basic elements */
	auto *replicator_layout = new QHBoxLayout;
	setLayoutMargins(replicator_layout);
	replicator_layout->addWidget(key_label);
	replicator_layout->addWidget(plus_button, 0, Qt::AlignLeft);
	replicator_layout->addWidget(minus_button, 0, Qt::AlignLeft);
	if (!no_spacers)
		replicator_layout->addSpacerItem(buildSpacer()); //keep widgets to the left
	addHelp(replicator_layout, options);

	/* layout of the basic elements plus children */
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(replicator_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //set the child
	container_->setVisible(false); //only visible when an item is selected
}

/**
 * @brief Parse options for a Replicator from XML.
 * @param[in] options XML node holding the Replicator.
 * @return True if all options were set successfully.
 */
bool Replicator::setOptions(const QDomNode &options)
{
	templ_ = options; //save a reference to the child XML node (shallow copy)
	this->setObjectName(getQtKey(section_ + Cst::sep + options.toElement().attribute("key")));
	return true;
}

/**
 * @brief Event listener for the plus button: replicate the child widget.
 * @details The child was saved as XML node, here this is parsed and built.
 */
void Replicator::replicate()
{
	element_counter_++;
	QDomNode node = prependParent(templ_); //prepend artificial parent node for recursion (runs through children)
	node.firstChildElement().setAttribute("replicate", "false"); //set node to normal element to be constructed
	QDomElement element = node.toElement();

	//recursively inject the element's number into the childrens' keys:
	substituteKeys(element, "#", QString::number(element_counter_));

	setUpdatesEnabled(false);
	Group *new_group = new Group(section_, "_replicator_item_" + key_);
	recursiveBuild(node, new_group, section_); //construct the children
	container_->addWidget(new_group);
	container_->setVisible(true);
	setUpdatesEnabled(true);
}

/**
 * @brief Event listener for the minus button: Remove the instance of the child widget created last.
 */
void Replicator::deleteLast()
{
	if (element_counter_ == 0)
		return; //no more widgets left
	if (element_counter_ == 1)
		container_->setVisible(false);
	auto *to_delete = qobject_cast<Group *>(container_->getLayout()->takeAt(element_counter_ - 1)->widget());
	if (to_delete) {
		to_delete->erase(); //delete the group's children
		delete to_delete; //delete the group itself
	}
	element_counter_--;
}


void Replicator::onPropertySet()
{
//	std::cout <<"Replicator onPropertySet: "<< this->property("ini_value").toString().toStdString()<<std::endl;
	const QString panel_to_add = this->property("ini_value").toString();
	replicate(); //TODO: transport number
}
