////////////////////////////////////////
///           NUMBER panel           ///
////////////////////////////////////////

#include "Number.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

#include "lib/tinyexpr.h"

#include <QDoubleSpinBox>
#include <QSpinBox>

#include <climits> //for the number panel limits

/**
 * @class Number
 * @brief Default constructor for a Number panel.
 * @details A number panel displays and manipulates a float or integer value.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Number panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Number::Number(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* number widget depending on which type of number to display */
	const QString format = options.toElement().attribute("format");
	if (format == "float" || format.isEmpty()) {
		number_element_ = new QDoubleSpinBox;
		mode_ = NR_DECIMAL;
	} else if (format == "integer" || format == "integer+") {
		number_element_ = new QSpinBox;
		mode_ = (format == "integer")? NR_INTEGER : NR_INTEGERPLUS;
	} else {
		topLog(tr("Unknown number format in XML file"), colors::getQColor("warning"));
		//TODO: abort
	}
	setPrimaryWidget(number_element_);
	auto *key_label = new Label(section_, "_number_label_" + key_, options, no_spacers, key_);
	number_element_->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	number_element_->setMinimumWidth(Cst::tiny);

	expression_element_ = new QLineEdit(this);
	expression_element_->hide();
	expression_element_->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	expression_element_->setMinimumWidth(Cst::tiny);
	connect(expression_element_, &QLineEdit::textChanged, this, &Number::checkStrValue);

	/* switch button and layout for number element plus button */
	switch_button_ = new QToolButton;
	connect(switch_button_, &QToolButton::toggled, this, &Number::switchToggle);
	switch_button_->setAutoRaise(true);
	switch_button_->setCheckable(true);
	switch_button_->setStyleSheet("QToolButton:checked {background-color: " +
	    colors::getQColor("lightblue").name() + "}");
	switch_button_->setIcon(QIcon("resources/icons/formula.png"));

	switcher_layout_ = new QHBoxLayout;
	switcher_layout_->addWidget(number_element_);
	if (options.toElement().attribute("notoggle") != "true")
		switcher_layout_->addWidget(switch_button_);

	/* layout of basic elements */
	auto *number_layout = new QHBoxLayout;
	setLayoutMargins(number_layout);
	number_layout->addWidget(key_label);
	number_layout->addLayout(switcher_layout_);
	if (!no_spacers)
		number_layout->addSpacerItem(buildSpacer()); //keep widgets to the left
	addHelp(number_layout, options);

	/* main layout */
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(number_layout);
	this->setLayout(layout);

	setOptions(options); //min, max, default, ...
}

/**
 * @brief Parse options for a Number panel from XML.
 * @param[in] options XML node holding the Number panel.
 * @return True if all options were set successfully.
 */
bool Number::setOptions(const QDomNode &options)
{
	const QString maximum = options.toElement().attribute("maximum");
	const QString minimum = options.toElement().attribute("minimum");
	const QString unit = options.toElement().attribute("unit");

	if (mode_ == NR_DECIMAL) {
		auto *spinbox = qobject_cast<QDoubleSpinBox *>(number_element_);
		bool success = true;
		/* minimum and maximum, choose whole range if they aren't set */
		double min = minimum.isEmpty()? std::numeric_limits<double>::min() : minimum.toDouble(&success);
		if (!success)
			topLog(tr("Could not parse minimum value for key ") + key_);
		double max = maximum.isEmpty()? std::numeric_limits<double>::max() : maximum.toDouble(&success);
		if (!success)
			topLog(tr("Could not parse maximum value for key ") + key_);
		/* unit */
		spinbox->setRange(min, max);
		if (!unit.isEmpty()) //for the space before the unit
			spinbox->setSuffix(" " + unit);
		connect(number_element_, SIGNAL(valueChanged(const double &)), this, SLOT(checkValue(const double &)));
	} else if (mode_ == NR_INTEGER || mode_ == NR_INTEGERPLUS) {
		auto *spinbox = qobject_cast<QSpinBox *>(number_element_);
		if (options.toElement().attribute("wrap") == "true") //circular wrapping when min/max is reached
			spinbox->setWrapping(true);
		/* minimum and maximum */
		bool success = true;
		int min = 0; //integer+
		if (mode_ == NR_INTEGER)
			min = minimum.isEmpty()? std::numeric_limits<int>::min() : minimum.toInt(&success);
		if (!success)
			topLog(tr("Could not parse minimum value for key ") + key_);
		int max = maximum.isEmpty()? std::numeric_limits<int>::max() : maximum.toInt(&success);
		if (!success)
			topLog(tr("Could not parse maximum value for key ") + key_);
		spinbox->setRange(min, max);
		/* unit */
		if (!unit.isEmpty())
			spinbox->setSuffix(" " + unit);
		connect(number_element_, SIGNAL(valueChanged(const int &)), this, SLOT(checkValue(const int &)));
	} //endif format

	return true; //TODO: some checks
}

void Number::checkValue(const double &to_check)
{
	auto *spinbox = static_cast<QDoubleSpinBox *>(number_element_);
	bool success = true;
	double value = this->property("ini_value").toDouble(&success);
	if (success) {
		if (value < spinbox->minimum())
			return;
		if (value > spinbox->maximum())
			return;
	}
	setDefaultPanelStyles(QString::number(to_check));
	setIniValue(to_check);
}

void Number::checkValue(const int &to_check)
{
	auto *spinbox = static_cast<QSpinBox *>(number_element_);
	bool success = true;
	int value = this->property("ini_value").toInt(&success);
	if (success) {
		if (value < spinbox->minimum()) {
			topLog(tr("Truncated integer value \"%1\" out of minimum range \"%2\" for INI key \"%3\"").arg(to_check).arg(spinbox->minimum()).arg(key_),
			    colors::getQColor("iniwarning"));
			value = spinbox->minimum();
		}
		if (value > spinbox->maximum()) {
			topLog(tr("Truncated integer value \"%1\" out of maximum range \"%2\" for INI key \"%3\"").arg(to_check).arg(spinbox->maximum()).arg(key_),
			    colors::getQColor("iniwarning"));
			value = spinbox->maximum();
		}
	}
	setDefaultPanelStyles(QString::number(to_check));
	setIniValue(to_check);
}

void Number::checkStrValue(const QString &str_check)
{
	//TODO: check variables / environment syntax
	static const QString regex_expression(R"(\${{(.+)}})");

	static const size_t idx_total = 0;
	static const size_t idx_expression = 1;

	static const QRegularExpression rex(regex_expression);
	const QRegularExpressionMatch match = rex.match(str_check);
	if (!str_check.isEmpty() && str_check == match.captured(idx_total)) {
		int status_code;
		te_interp(match.captured(idx_expression).toStdString().c_str(), &status_code);
		if (status_code != 0) { //it's a valid arithmetic expression
			setPanelStyle(FAULTY);
		} else {
			setPanelStyle(VALID);
			setIniValue(str_check);
		}
	} else {
		setDefaultPanelStyles(str_check);
		setIniValue(str_check);
	}
}

bool Number::isExpression(const QString &expression) const
{
	if (expression.mid(0, 2) == "${" && expression.mid(expression.length()-1) == "}") //.back() is in Qt 5.10
		return true;
	else
		return false;
}

void Number::onPropertySet()
{
	const QString str_value = this->property("ini_value").toString();

	if (isExpression(str_value)) {
		expression_element_->setText(str_value);
		switch_button_->setChecked(true);
		return;
	}

	if (auto *spinbox = qobject_cast<QSpinBox *>(number_element_)) { //integer
		int ival = str_value.toInt();
		if (ival == 0)
			emit checkValue(ival); //if the default isn't changed from zero then nothing is emitted
		else
			spinbox->setValue(str_value.toInt());
	} else if (auto *spinbox = qobject_cast<QDoubleSpinBox *>(number_element_)) { //floating point
		double dval = str_value.toDouble();
		if (qFuzzyIsNull(dval)) //fuzzy against warnings
			emit checkValue(dval);
		else
			spinbox->setValue(str_value.toDouble());
	}
}

void Number::switchToggle(bool checked)
{
	if (checked) {
		switcher_layout_->replaceWidget(number_element_, expression_element_);
		number_element_->hide();
		expression_element_->show();
		setPrimaryWidget(expression_element_);
		setIniValue(expression_element_->text());
		setDefaultPanelStyles(expression_element_->text());
	} else {
		switcher_layout_->replaceWidget(expression_element_, number_element_);
		expression_element_->hide();
		number_element_->show();
		setPrimaryWidget(number_element_);
		if (mode_ == NR_DECIMAL) {
			setIniValue(static_cast<QDoubleSpinBox *>(number_element_)->value());
			setDefaultPanelStyles(QString::number(static_cast<QDoubleSpinBox *>(number_element_)->value()));
		} else {
			setIniValue(static_cast<QSpinBox *>(number_element_)->value());
			setDefaultPanelStyles(QString::number(static_cast<QSpinBox *>(number_element_)->value()));
		}
	}
}
