////////////////////////////////////////
///        HORIZONTAL raster         ///
////////////////////////////////////////

#include "HorizontalPanel.h"
#include "src/main/inishell.h"

/**
 * @class HorizontalPanel
 * @brief Default constructor for a HorizontalPanel.
 * @details A HorizontalPanel is a simple layout that organizes child widgets horizontally. The children
 * are given enclosed in <option> tags in the XML file.
 * @param[in] section The INI section is set for style targeting.
 * @param[in] key INI key is used for an optional label and ignored otherwise.
 * @param[in] options XML node responsible for this panel with all desired children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
HorizontalPanel::HorizontalPanel(const QString &section, const QString &key, const QDomNode &options,
    const bool &no_spacers, QWidget *parent) : Atomic(section, key, parent)
{
	horizontal_layout_ = new QHBoxLayout; //only holds a horizontal layout for child panels
	setLayoutMargins(horizontal_layout_);
	this->setLayout(horizontal_layout_);

	setOptions(options, no_spacers); //construct children
	addHelp(horizontal_layout_, options); //children and the panel can both have help texts
}

/**
 * @brief Parse options for a HorizontalPanel from XML.
 * @param[in] options XML node holding the HorizontalPanel.
 * @return True if all options were set successfully.
 */
bool HorizontalPanel::setOptions(const QDomNode &options, const bool &no_spacers)
{
	if (!key_.isEmpty()) { //display caption
		auto *key_label = new Label(section_, "_horizontal_label_" + key_, options, true, key_);
		horizontal_layout_->addWidget(key_label, 0, Qt::AlignLeft | Qt::AlignCenter);
	}
	/* build all children */
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		substituteKeys(op, "@", this->key_);
		auto *item_group = new Group(section_, "_horizontal_itemgroup_" + key_);
		recursiveBuild(op, item_group, section_, true); //recursive build with horizontal space savings
		horizontal_layout_->addWidget(item_group, 0, Qt::AlignVCenter | Qt::AlignLeft);
	}
	if (!no_spacers)
		horizontal_layout_->addSpacerItem(buildSpacer()); //keep widgets to the left
	return true;
}
