////////////////////////////////////////
///         HELPTEXT element         ///
////////////////////////////////////////

#include "Helptext.h"
#include "src/main/colors.h"
#include "src/main/constants.h"

/**
 * @class Helptext
 * @brief Default constructor for a Helptext.
 * @details A Helptext is a styled label with properties set to display it on the far right of a layout.
 * The help text can either be displayed with a fixed width for a uniform look.
 * Or it can be displayed in a single line, in which case it is shown right next to the panel it describes
 * and extens as far as the text goes, viewable with the main horizontal scroll bar.
 * @param[in] text The help text.
 * @param[in] single_line If set, do not try to align the text in multiple columns. Rather, display it
 * in a single line and use the main scroll bars for it.
 * @param[in] parent The parent widget.
 */
Helptext::Helptext(const QString &text, const bool &single_line, QWidget *parent) : QLabel(parent)
{
	this->setTextFormat(Qt::RichText);
	this->setTextInteractionFlags(Qt::TextBrowserInteraction);
	this->setOpenExternalLinks(true); //clickable links
	QFont font(this->font());
	font.setPointSize(Cst::help_font_size); //smaller font
	this->setFont(font);
	QPalette label_palette(this->palette()); //custom color
	label_palette.setColor(QPalette::WindowText, colors::getQColor("helptext"));
	this->setPalette(label_palette);

	if (!single_line) { //no space to preceding elements, users scroll to the end
		this->setWordWrap(true); //multi line
		const QFontMetrics font_metrics(font);
		const int text_width = font_metrics.boundingRect(text).width(); //real size of text rectangle on screen
		//Texts that fit in a single line are right-aligned so they sit at the very right rather than
		//on the left side of a box that is help_width big (thus creating margins):
		if (text_width <= Cst::help_width)
			this->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
		//standard Helptext width, or the text width if it is smaller than that:
		this->setFixedWidth(getMinTextSize(text, Cst::help_width));
	}
	this->setText(text);
}

/**
 * @brief Set a new help text.
 * @param[in] text The new help text to display.
 */
void Helptext::updateText(const QString &text)
{
	this->setText(text);
}

/**
 * @brief Compare the standard Helptext width with the width of the text.
 * @details If the width of the text is smaller than the standard help text width then this
 * function returns the smaller text width. This way the element floats freely without
 * unnecessary margins.
 * @param[in] text Text that will be displayed.
 * @param[in] min_width
 * @return The fixed width to use for this Helptext.
 */
int Helptext::getMinTextSize(const QString &text, const int &standard_width)
{
	const QFontMetrics font_metrics(this->font());
	const int text_width = font_metrics.boundingRect(text).width(); //horizontal pixels
	if (text_width < standard_width)
		return text_width + Cst::label_padding; //tiny bit of room to not wrap unexpectedly
	else
		return standard_width;
}
