////////////////////////////////////////
///         GROUPING element         ///
////////////////////////////////////////

#include "Group.h"
#include "src/main/colors.h"

/**
 * @class Group
 * @brief Default Group constructor
 * @details Group is the central grouping element which the GUI building recursion works on.
 * Essentially it is a wrapper for QGroupBox.
 * The main program tab holds Groups in which the panels go. If the panels feature
 * child elements themselves they too own a group in which they are put.
 * @param[in] section Name of the INI section the group corresponds to. This could be used
 * to style the groups differently depending on the section, but for now it is unused. This
 * parameter does not affect the childrens' settings.
 * @param[in] key (INI) key for the group. Does not affect the childrens' keys.
 * @param[in] has_border Optional border around the group.
 * @param[in] grid_layout Elements are not placed vertically (default) but in a grid layout.
 * @param[in] is_frame Act as a frame with border and title.
 * @param[in] caption If the group is a frame the caption is displayed as title.
 * @param[in] parent The parent widget.
 */
Group::Group(const QString &section, const QString &key, const bool &has_border,
	    const bool &grid_layout, const bool &is_frame, const QString &caption, QWidget *parent)
	    : Atomic(section, key, parent)
{
	box_ = new QGroupBox(is_frame? caption : QString()); //all children go here, only frame can have a title
	setPrimaryWidget(box_);
	if (grid_layout)
		layout_ = new QGridLayout; //mainly for the grid panel
	else
		layout_ = new QVBoxLayout; //frame is always this
	box_->setLayout(layout_); //we can add widgets at any time

	if (!is_frame) { //normal group
		if (!has_border) {
			box_->setStyleSheet("QGroupBox#_primary_" + getQtKey(getId()) + " {border: none; margin-top: 0px}");
		} else {
			box_->setStyleSheet("QGroupBox#_primary_" + getQtKey(getId()) +" {border: 1px solid " +
			    colors::getQColor("groupborder").name() + "; border-radius: 6px;}");
			setLayoutMargins(layout_);
		}
	} else { //it's a frame
		box_->setStyleSheet(" \
			    QGroupBox#_primary_" + getQtKey(getId()) + "  {font: bold; border: 2px solid " +
			    colors::getQColor("frameborder").name() + "; border-radius: 6px; margin-top: 8px; color: " +
			    colors::getQColor("frametitle").name() + ";} QGroupBox::title#_primary_" + getQtKey(getId()) +
			    " {subcontrol-origin: margin; left: 17px; padding: 0px 5px 0px 5px; \
			}");
		box_->layout()->setContentsMargins(Cst::frame_horizontal_margins, Cst::frame_vertical_margins,
		    Cst::frame_horizontal_margins, Cst::frame_vertical_margins); //a little room for the frame
	}

	auto *main_layout = new QVBoxLayout; //layout for the derived class to assign the box_ to it
	setLayoutMargins(main_layout); //tighter placements
	main_layout->addWidget(box_, 0, Qt::AlignTop); //alignment important when we hide/show children
	this->setLayout(main_layout);
}

/**
 * @brief Add a child widget to the group's vertical layout.
 * @param[in] widget The widget to add.
 */
void Group::addWidget(QWidget *widget)
{
	auto *layout = qobject_cast<QVBoxLayout *>(layout_);
	layout->addWidget(widget, 0, Qt::AlignTop); //alignment needed after we hide/show groups
}

/**
 * @brief Add a child widget to the group's grid layout.
 * @details This is called by the GridPanel to add widgets in a raster.
 * @param[in] widget The widget to add.
 * @param[in] row Row position of the widget (starts at 0).
 * @param[in] column Column position of the widget (starts at 0).
 * @param[in] rowSpan The widget spans this many rows (default: 1).
 * @param[in] columnSpan The widget spans this many columns (default: 1).
 * @param[in] alignment Alignment of the widget within the raster point.
 */
void Group::addWidget(QWidget *widget, int row, int column, int rowSpan, int columnSpan,
    Qt::Alignment alignment)
{
	auto *layout = qobject_cast<QGridLayout *>(layout_); //TODO: safechecks
	layout->addWidget(widget, row, column, rowSpan, columnSpan, alignment);
}

/**
 * @brief Delete all child panels of the group.
 * @details The group itself is not deleted, but the container is rendered useless.
 */
void Group::erase()
{
	setUpdatesEnabled(false); //disable painting in case we have a lot of content
	delete box_;
	setUpdatesEnabled(true);
}

/**
 * @brief Retrieve the number of child panels.
 * @return The number of child panels.
 */
int Group::count() const
{
	return (this->getLayout()->count());
}

/**
 * @brief Check if the group has child panels.
 * @return True if there is at least one child.
 */
bool Group::isEmpty() const
{
	return (this->count() == 0);
}

/**
 * @brief Check if the group has visible child groups.
 * @details If a child group is visible but has no visible content, it is still considered visible.
 * This is enough for Checklist and Choice which hide the groups sufficiently.
 * @return True if at least one child group is visible.
 */
bool Group::hasVisibleChildren() const
{
	QList<Group *> list = box_->findChildren<Group *>();
	for (auto &widget : list) {
		if (widget->isVisible())
			return true;
	}
	return false;
}
