////////////////////////////////////////
///          DROPDOWN panel          ///
////////////////////////////////////////

#include "Dropdown.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

/**
 * @class Dropdown
 * @brief Default constructor for a Dropdown panel.
 * @details A Dropdown panel shows a dropdown menu where additional options pop up depending on the
 * selected menu item.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Choice panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Dropdown::Dropdown(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and combo box */
	dropdown_ = new QComboBox;
	setPrimaryWidget(dropdown_);
	dropdown_->setMinimumWidth(Cst::tiny); //no tiny elements
	connect(dropdown_, QOverload<int>::of(&QComboBox::currentIndexChanged), this, &Dropdown::itemChanged);

	/* layout of the basic elements */
	auto *dropdown_layout = new QHBoxLayout;
	setLayoutMargins(dropdown_layout);
	if (!key.isEmpty()) {
		auto *key_label = new Label(section_, key_, options, no_spacers, key_);
		dropdown_layout->addWidget(key_label, 0, Qt::AlignLeft);
	}
	dropdown_layout->addWidget(dropdown_, 0, Qt::AlignLeft);
	//add a big enough spacer to the right to keep the buttons to the left (unless it's a horizontal layout):
	if (!no_spacers)
		dropdown_layout->addSpacerItem(buildSpacer());
	addHelp(dropdown_layout, options);

	/* layout of the basic elements plus children */
	container_ = new Group(section, "_dropdown_" + key_, true);
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(dropdown_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //construct child panels
	emit itemChanged(0); //select first option
}

/**
 * @brief Parse options for a Dropdown panel from XML.
 * @param[in] options XML node holding the Dropdown panel.
 * @return True if all options were set successfully.
 */
bool Dropdown::setOptions(const QDomNode &options)
{
	if (options.toElement().attribute("optional") != "false") {
		dropdown_->addItem("");
		Group *dummy_group = new Group(section_, "_dummy_group_" + key_, false);
		container_->addWidget(dummy_group);
	}
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		substituteKeys(op, "@", this->key_ + Cst::sep);
		/* add childrens' names to dropdown and build them */
		dropdown_->addItem(op.attribute("value"));
		auto *item_group = new Group(section_, "_item_dropdown_" + key_); //group all elements of this option together
		recursiveBuild(op, item_group, section_);
		container_->addWidget(item_group);
	}
	return true;
}

/**
 * @brief Event listener for when a dropdown menu item is selected.
 * @details This function shows/hides the children corresponding to the selected dropdown item.
 * @param[in] index Index/number of the selected item.
 */
void Dropdown::itemChanged(int index)
{
	QLayout *group_layout = container_->getLayout();
	for (int ii = 0; ii < group_layout->count(); ++ii) {
		//find the group to display and hide all others:
		auto *group_item = qobject_cast<Group *>(group_layout->itemAt(ii)->widget());
		group_item->setVisible(index == ii);
		if (ii == index) //hide the container if the displayed group is empty
			container_->setVisible(!group_item->isEmpty());
		/*
		 * Finally, set a flag in the INI parser for the keys that are currently hidden / inactive.
		 * This, or something similar, is necessary because if we only checked whether a panel is visible
		 * when setting INI keys on initialization we would miss default values with the current flowchart.
		 */
		QList<Atomic *> widget_list = group_item->findChildren<Atomic *>();
		for (int jj = 0; jj < widget_list.count(); ++jj)
			widget_list.at(jj)->setIniActive(index == ii);
	}

	const QString dropdown_text(dropdown_->currentText());
	setDefaultPanelStyles(dropdown_text);
	setIniValue(dropdown_text);
}

void Dropdown::onPropertySet()
{
	const QString text_to_set = this->property("ini_value").toString();
	for (int ii = 0; ii < dropdown_->count(); ++ii) {
		if (dropdown_->itemText(ii).toLower() == text_to_set.toLower()) {
			dropdown_->setCurrentIndex(ii);
			return;
		}
	}
	topLog(tr("Alternative value \"%1\" could not be set from INI file for key \"%2\": no such option specified in XML file").arg(
	    text_to_set).arg(key_), colors::getQColor("iniwarning"));
}
