////////////////////////////////////////
///           CHOICE panel           ///
////////////////////////////////////////

#include "Choice.h"
#include "Label.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

/**
 * @class Choice
 * @brief Default constructor for a Choice panel.
 * @details A choice panel shows a list of checkboxes, each of which controls showing/hiding of
 * additional options.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Choice panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Choice::Choice(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	const QDomElement element(options.toElement());

	//grouping element of the list of checkboxes:
	checkbox_container_ = new Group(section, "_choice_checkbox_container_" + key_, false, true); //grid layout
	setPrimaryWidget(checkbox_container_);
	//grouping element for all children:
	child_container_ = new Group(section, "_choice_child_container_" + key_); //vertical layout
	child_container_->setVisible(false);
	auto *key_label = new Label(section_, "_choice_label_" + key_, options, no_spacers, key_);
	checkbox_container_->addWidget(key_label, 0, 0, 1, 1);

	/* layout for checkboxes and children together */
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addWidget(checkbox_container_);
	layout->addWidget(child_container_);
	this->setLayout(layout);

	setOptions(options); //build children
}

/**
 * @brief Parse options for a Choice panel from XML.
 * @param[in] options XML node holding the Choice panel.
 * @return True if all options were set successfully.
 */
bool Choice::setOptions(const QDomNode &options)
{
	int counter = 0;
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		auto *checkbox = new QCheckBox(op.attribute("value"));
		checkbox->setMinimumHeight(30); //GUI smoothing
		//connect to lambda function to emit current index (modern style signal mapping):
		connect(checkbox, &QCheckBox::stateChanged, this, [=] { changedState(counter); });
		checkbox_container_->addWidget(checkbox, counter, 1);

		/* help text */
		QString helptext = op.firstChildElement("help").text();
		if (helptext.isEmpty()) //same as addHelp but for a certain grid position
			helptext = op.attribute("help");
		auto *help = new Helptext(helptext, false); //if this is made optional account for it in onPropertySet's ... -1!
		checkbox_container_->addWidget(help, counter, 2, 1, 1, Qt::AlignRight);

		/* child elements of this checkbox */
		auto *item_group = new Group(section_, "_item_choice_" + key_);
		recursiveBuild(op, item_group, section_);
		item_group->setVisible(false);
		child_container_->addWidget(item_group);
		counter++;
	}
	return true;
}

/**
 * @brief Event listener for when a single checkbox is checked/unchecked.
 * @details This function shows/hides child elements when a checkbox changes.
 * @param[in] index The index/row of the clicked item.
 */
void Choice::changedState(int index)
{
	QLayout *group_layout = child_container_->getLayout(); //get item number 'index' from the child group's layout
	auto *item_group = qobject_cast<Group *>(group_layout->itemAt(index)->widget());
	auto *clicked_box =
	    qobject_cast<QCheckBox *>(checkbox_container_->getGridLayout()->itemAtPosition(index, 1)->widget());
	setUpdatesEnabled(false);
	//show the clicked child's group on item check if it's not empty:
	item_group->setVisible(clicked_box->checkState() == Qt::Checked && !item_group->isEmpty());
	child_container_->setVisible(true);
	/*
	 * Set a flag in the INI parser for the keys that are currently hidden / inactive.
	 * This, or something similar, is necessary because if we only checked whether a panel is visible
	 * when setting INI keys on initialization we would miss default values with the current flowchart.
	 */
	QList<Atomic *> widget_list = item_group->findChildren<Atomic *>();
	for (int ii = 0; ii < widget_list.count(); ++ii)
		widget_list.at(ii)->setIniActive(clicked_box->checkState() == Qt::Checked);
	setUpdatesEnabled(true);

	QString list_values = ""; //important for the INI parser that it's not Null
	for (int ii = 0; ii < checkbox_container_->getGridLayout()->rowCount(); ++ii) {
		auto *checkbox = qobject_cast<QCheckBox *>(checkbox_container_->getGridLayout()->itemAtPosition(ii, 1)->widget());
		if (checkbox->checkState() == Qt::Checked)
			list_values += checkbox->text() + " ";
	}
	setDefaultPanelStyles(list_values);
	setIniValue(list_values);
}

void Choice::onPropertySet()
{
	const QString values = this->property("ini_value").toString();
	const QStringList value_list = values.split(QRegExp("\\s+"), QString::SkipEmptyParts);

	for (int ii = 0; ii < value_list.size(); ++ii) {
		bool found_option_to_set = false;
		for (int jj = 0; jj < checkbox_container_->getGridLayout()->rowCount(); ++jj) {
			auto *checkbox = qobject_cast<QCheckBox *>(checkbox_container_->getGridLayout()->itemAtPosition(jj, 1)->widget());
			if (checkbox->text().toLower() == value_list.at(ii).toLower()) {
				checkbox->setCheckState(Qt::Checked);
				found_option_to_set = true;
				break;
			}
		}
		if (!found_option_to_set)
			topLog(tr("Choice item \"%1\" could not be set from INI file for key \"%2\": no such option specified in XML file").arg(
			    value_list.at(ii)).arg(key_), colors::getQColor("iniwarning"));
	} //endfor ii
}
