/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Checklist.h"
#include "Label.h"
#include "src/main/colors.h"
#include "src/main/inishell.h"

#include <QStringList>

/**
 * @class Checklist
 * @brief Default constructor for a Checklist panel.
 * @details A Checklist shows a checkable list where each list item can have arbitrary children
 * which are displayed below if the item is checked and hidden if it's not.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Checklist.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Checklist::Checklist(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and list */
	auto *key_label = new Label(section_, "_checklist_label_" + key_, options, no_spacers, key_);
	list_ = new QListWidget;
	setPrimaryWidget(list_);
	connect(list_, &QListWidget::itemClicked, this, &Checklist::listClick);

	/* layout of the basic elements */
	checklist_layout_ = new QHBoxLayout;
	setLayoutMargins(checklist_layout_);
	checklist_layout_->addWidget(key_label, 0, Qt::AlignTop);
	checklist_layout_->addWidget(list_);
	main_help_ = addHelp(checklist_layout_, options, true);
	main_help_->setFixedWidth(Cst::help_width);
	main_help_->setProperty("main_help", options.firstChildElement("help").text());

	/* layout of basic elements plus children */
	container_ = new Group(section, "_checklist_container_" + key_); //generic name that could be used for stylesheets
	container_->setVisible(false); //container for children activates only when clicked (saves space)
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(checklist_layout_);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //fill list items
}

/**
 * @brief Parse options for a Checklist from XML.
 * @param[in] options XML node holding the Checklist.
 * @return True if all options were set successfully.
 */
bool Checklist::setOptions(const QDomNode &options)
{
	QStringList item_strings;
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		//add the option value as list item:
		auto *item = new QListWidgetItem(op.attribute("value"), list_);
		item_strings.push_back(op.attribute("value"));
		item->setFlags(item->flags() & ~Qt::ItemIsUserCheckable); //not checkable because we do it on row click
		item->setCheckState(Qt::Unchecked);
		item->setToolTip(op.attribute("help"));
		item->setData(Qt::UserRole, op.firstChildElement("help").text());

		//construct the children of this option:
		auto *item_group = new Group(section_, "_checklist_itemgroup_" + key_); //group all elements of this option together
		recursiveBuild(op, item_group, section_);
		container_->addWidget(item_group);
		item_group->setVisible(false); //becomes visible when checked
	}
	//this many rows should be visible (with some padding):
	list_->setFixedHeight(list_->sizeHintForRow(0) * Cst::nr_items_visible +
	    Cst::checklist_safety_padding_vertical);
	list_->setMinimumWidth(getMaximumTextWidth(item_strings, Cst::maximum_checklist_width) +
	    Cst::checklist_safety_padding_horizontal);
	return true;
}

/**
 * @brief Event handler for clicks in the list widget.
 * @details This handles showing and hiding children belonging to list items.
 * @param[in] item Item that was clicked.
 */
void Checklist::listClick(QListWidgetItem *item)
{
	QLayout *group_layout = container_->getLayout();
	auto *item_group = qobject_cast<Group*>(group_layout->itemAt(list_->currentRow())->widget());

	//click in line is enough to check/uncheck:
	if (item->checkState() == Qt::Unchecked) { //now it's CHECKED
		item->setCheckState(Qt::Checked);
		//show/hide children of list option that has been clicked:
		item_group->setVisible(!item_group->isEmpty());
		if (!item_group->isEmpty())
			container_->setVisible(true);
		main_help_->updateText(item->data(Qt::UserRole).toString());
	} else { //now it's UNCHECKED
		setUpdatesEnabled(false); //we only experience flickering when hiding
		item->setCheckState(Qt::Unchecked);
		item_group->setVisible(false);
	}

	QString list_values = ""; //it's important for the INI parser that this is not Null
	bool one_visible = false;
	bool all_unchecked = true;
	for (int ii = 0; ii < list_->count(); ++ii) { //hide container if empty (a few blank pixels):
		if (list_->item(ii)->checkState() == Qt::Checked) {
			all_unchecked = false;
			if (!static_cast<Group *>(group_layout->itemAt(ii)->widget())->isEmpty())
				one_visible = true;
			list_values += list_->item(ii)->text() + " ";
		}
	}
	container_->setVisible(one_visible);
	if (all_unchecked)
		main_help_->updateText(main_help_->property("main_help").toString());
	else
		list_values.chop(1); //remove trailing space


	/*
	 * Set a flag in the INI parser for the keys that are currently hidden / inactive.
	 * This, or something similar, is necessary because if we only checked whether a panel is visible
	 * when setting INI keys on initialization we would miss default values with the current flowchart.
	 */
	QList<Atomic *> widget_list = item_group->findChildren<Atomic *>();
	for (int ii = 0; ii < widget_list.count(); ++ii)
		widget_list.at(ii)->setIniActive(item->checkState() == Qt::Checked);

	setDefaultPanelStyles(list_values);
	setIniValue(list_values);
	setBufferedUpdatesEnabled();
}

void Checklist::onPropertySet()
{
	const QString values = this->property("ini_value").toString();
	const QStringList value_list = values.split(QRegExp("\\s+"), QString::SkipEmptyParts);

	for (int ii = 0; ii < list_->count(); ++ii) { //clear the list
		if (list_->item(ii)->checkState() == Qt::Checked) {
			list_->setCurrentRow(ii);
			emit listClick(list_->item(ii));
		}
	}

	for (int ii = 0; ii < value_list.size(); ++ii) {
		bool found_option_to_set = false;
		for (int jj = 0; jj < list_->count(); ++jj) {
			if (list_->item(jj)->text().toLower() == value_list.at(ii).toLower()) {
				list_->setCurrentRow(jj); //else we will get NULL when we fetch the item
				emit listClick(list_->item(jj)); //to set the default value
				found_option_to_set = true;
				break;
			}
		}
		if (!found_option_to_set)
			topLog(tr("Checklist item \"%1\" could not be set from INI file for key \"%2\": no such option specified in XML file").arg(
			    value_list.at(ii)).arg(key_), colors::getQColor("iniwarning"));
	} //endfor ii
}
