/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Atomic.h"
#include "src/main/inishell.h"

#include <QCryptographicHash>
#include <QFontMetrics>

#include <utility> //for modern constructor move semantics

/**
 * @class Atomic
 * @brief Base class of most panels.
 * @details The panels inherit from this class which provides some common functionality
 * like handling the object IDs.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this panel.
 * @param[in] parent The parent widget.
 */
Atomic::Atomic(QString section, QString key, QWidget *parent)
    : QWidget(parent), section_(std::move(section)), key_(std::move(key))
{
	ini_ = getMainWindow()->getIni();
}

/**
 * @brief Check if a panel value is mandatory or currently at the default value.
 * @param[in] in_value The current value of the panel.
 */
void Atomic::setDefaultPanelStyles(const QString &in_value)
{
	setPanelStyle(FAULTY, false); //first we disable temporary styles
	setPanelStyle(VALID, false);
	const bool is_default = (in_value == this->property("default_value"));
	setPanelStyle(DEFAULT, is_default);
	if (this->property("is_mandatory").toBool())
		setPanelStyle(MANDATORY, in_value.isEmpty());
}

/**
 * @brief Enable or disable an INI key for file output.
 * @details Cf. notes on INIParser::setActive().
 * @param[in] is_active If true, the key is written out when saving an INI file.
 */
void Atomic::setIniActive(const bool& is_active)
{
	ini_->setActive(section_, key_, is_active);
}

/**
 * @brief Get an alphanumeric key from an aribtrary one.
 * @details Unfortunately, our arg1::arg2 syntax runs into Qt's sub-controls syntax, but the targeting
 * with stylesheets only allows _ and - anyway so we hash the ID.
 * @param[in] ini_key Key as given in the XML
 * @return A key without special chars.
 */
QString Atomic::getQtKey(const QString &ini_key)
{
#ifndef DEBUG
	QString hashed(QCryptographicHash::hash((ini_key.toLower().toLocal8Bit()), QCryptographicHash::Md5).toHex());
#else //more readable names in debug mode, bad luck needed to be troublesome:
	QString hashed(ini_key);
	hashed.replace(QRegularExpression(R"([^\w-])"), "-"); //alternative
#endif
	return hashed;
}

/**
 * @brief Set a panel's primary widget.
 * @details The widget pointed to this way is responsible for actually changing a value and
 * is used for highlighting purposes.
 * @param primary_widget The panel's main widget.
 */
void Atomic::setPrimaryWidget(QWidget *primary_widget, const bool &set_object_name)
{
	primary_widget_ = primary_widget;
	primary_widget->setObjectName("_primary_" + getQtKey(getId()));
	if (set_object_name) //template panels may want to do this themselves (handling substitutions)
		this->setObjectName(getQtKey(getId()));
	QObject *const property_watcher = new PropertyWatcher(primary_widget);
	this->connect(property_watcher, SIGNAL(changedValue()), SLOT(onPropertySet()));
	this->installEventFilter(property_watcher);
	setDefaultPanelStyles(this->property("ini_value").toString());
}

/**
 * @brief Set a property indicating that the value this panel controls is defaulted or mandatory,
 * or to be highlighted in a different way.
 * @param set Set style on/off.
 */
void Atomic::setPanelStyle(const PanelStyle &style, const bool &set)
{
	if (primary_widget_ == nullptr) //e. g. horizontal panel
		return;
	QString style_string;
	switch (style) {
	case MANDATORY:
		style_string = "mandatory";
		break;
	case DEFAULT:
		style_string = "shows_default";
		break;
	case FAULTY:
		style_string = "faulty";
		break;
	case VALID:
		style_string = "valid";
	}
	this->primary_widget_->setProperty(style_string.toLocal8Bit(), set? "true" : "false");
	this->style()->unpolish(primary_widget_); //if a property is set dynamically, we might have to refresh
	this->style()->polish(primary_widget_);
} //https://wiki.qt.io/Technical_FAQ#How_can_my_stylesheet_account_for_custom_properties.3F

/**
 * @brief Substitute values in keys and texts.
 * @details This is for child elements that inherit a property from its parent which should be
 * displayed (e. g. "TA" in "TA::FILTER1 = ..."). The function traverses through the whole child
 * tree recursively.
 * @param[in] parent_element Parenting XML node holding the desired values.
 * @param[in] replace String to replace.
 * @param[in] replace_with Text to replace the string with.
 */
void Atomic::substituteKeys(QDomElement &parent_element, const QString &replace, const QString &replace_with)
{
	for (QDomElement element = parent_element.firstChildElement(); !element.isNull(); element = element.nextSiblingElement()) {
		QString key(element.attribute("key"));
		element.setAttribute("key", key.replace(replace, replace_with));
		QString text(element.attribute("caption"));
		element.setAttribute("caption", text.replace(replace, replace_with)); //for labels
		substituteKeys(element, replace, replace_with);
	}
}

/**
 * @brief Build a standard spacer item for widget positioning.
 * @details This is the only little workaround we use for our layouts. The scroll areas are allowed
 * to resize everything, and sometimes they do so aggressively; for example, they ignore left-aligned
 * elements and move them to the right if the screen gets very big. If on the other hand we try to
 * adjust the sizes of all child widgets manually via .adjustSize() update(), etc., then we run into
 * troubles that require far uglier hacks if they can be solved at all. It has proven best to try not
 * to meddle with the layout manager.
 * The solution here is to add a huge spacer to a list of left-aligned widgets if we want to keep all of
 * them a fixed (small) size.
 * @return Spacer element that can be added to a layout.
 */
QSpacerItem * Atomic::buildSpacer()
{
	return new QSpacerItem(getMainWindow()->width() * 2, 0, QSizePolicy::Maximum); //huge spacer
}

/**
 * @brief Set margins of a layout.
 * @details This controls how much space there is between widgets, i. e. the spacing in the main GUI.
 * @param [in] layout Any layout.
 */
void Atomic::setLayoutMargins(QLayout *layout)
{
	//                  left, top, right, bottom
	layout->setContentsMargins(2, 1, 2, 1); //with our nested groups we want to keep them tight
}

/**
 * @brief Convenience call to add a Helptext object to the end of a vertical layout.
 * @details <help> elements are displayed in a Helptext, "help" attributes in the tooltip.
 * @param[in] layout Layout to add the Helptext to.
 * @param[in] options Parent XML node controlling the appearance of the help text.
 * @param[in] force Add a Helptext even if it's empty (used by panels that can change the text).
 */
Helptext * Atomic::addHelp(QHBoxLayout *layout, const QDomNode &options, const bool &force)
{
	const QDomElement help_element(options.firstChildElement("help")); //dedicated <help> tag if there is one
	const bool single_line = (help_element.attribute("nowrap") == "true"); //single line extending the scroll bar
	const QString helptext(help_element.text());
	if (primary_widget_ != nullptr) {
		const QString inline_help(help_element.attribute("help")); //help in attribute as opposed to element
		primary_widget_->setToolTip(inline_help);
	}
	if (force || !helptext.isEmpty()) {
		auto *help = new Helptext(helptext, single_line);
		layout->addWidget(help, 0, Qt::AlignRight);
		return help;
	}
	return nullptr;
}

/**
 * @brief Call setUpdatesEnabled() with a tiny delay
 * @details There is GUI flickering when hiding widgets, even for simple ones. It is occasional, but
 * sometimes quite annoying. Disabling and re-enabling the painting is not enough, but firing a timer
 * to do so helps quite a bit. This is used by panels that can show/hide child panels.
 */
void Atomic::setBufferedUpdatesEnabled()
{
	QTimer enable_timer;
	enable_timer.singleShot(0.5, this, &Atomic::onTimerBufferedUpdatesEnabled);
}

/**
 * @brief Find the widest text in a list.
 * @details This function calculates the on-screen width of a list of texts and returns the
 * greatest one, optionally capping at a fixed value.
 * @param[in] text_list List of texts to check.
 * @param[in] element_max_width Maximum allowed width to return.
 * @return The maximum text width, or the hard set limit.
 */
int Atomic::getMaximumTextWidth(const QStringList &text_list, const int &element_max_width)
{
	const QFontMetrics font_metrics(this->font());
	int max_width = 0;
	for (auto &text : text_list) {
		const int text_width = font_metrics.boundingRect(text).width();
		if (text_width > max_width)
			max_width = text_width;
	}
	if (max_width > element_max_width)
		max_width = element_max_width;
	return max_width;
}

/**
 * @brief Event handler for property changes.
 * @details Suitable panels override this to react to changes of the INI value from outside.
 */
void Atomic::onPropertySet()
{
	//Ignore this signal for unsuitable panels. Suitable ones have their own implementation.
}

/**
 * @brief Convert a number to string and pass to INI setter.
 * @param[in] value The integer to convert.
 */
void Atomic::setIniValue(const int &value)
{
	setIniValue(QString::number(value));
}

/**
 * @brief Convert a number to string and pass to INI setter.
 * @param[in] value The double value to convert.
 */
void Atomic::setIniValue(const double &value)
{
	setIniValue(QString::number(value));
}

/**
 * @brief Atomic::setIniValue
 * @details Gets called after range checks have been performed, i. e. when we
 * want to propagate the change to the INI. It is called by changes to:
 *  - Values through users entering in the GUI
 *  - Default values in the XML
 *  - Values from an INI file
 * @param[in] value Set the value for a key in the INIParser.
 */
void Atomic::setIniValue(const QString &value)
{
	ini_->set(section_, key_, value);
}

/**
 * @brief Re-enable GUI painting.
 * @details Cf. notes in setBufferedUpdatesEnabled().
 */
void Atomic::onTimerBufferedUpdatesEnabled()
{
	setUpdatesEnabled(true);
}
