/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The main window that is displayed on startup.
 * 2019-10
 */

#include "MainWindow.h"
#include "src/gui/AboutWindow.h"
#include "src/gui/Settings.h"
#include "src/main/colors.h"
#include "src/main/Error.h"
#include "src/main/dimensions.h"
#include "src/main/INIParser.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QApplication>
#include <QGroupBox>
#include <QMenuBar>
#include <QRegularExpression>
#include <QSpacerItem>
#include <QStatusBar>
#include <QSysInfo>
#include <QToolBar>

#ifdef DEBUG
	#include <QSpinBox>
	#include <iostream>
	#include <sstream>
#endif

/**
 * @class MainWindow
 * @brief Constructor for the main window.
 * @param[in] xml_settings Settings for the static INIshell GUI.
 * @param[in] parent Unused since the main window is kept in scope by the entry point function.
 */
MainWindow::MainWindow(QDomDocument &xml_settings, QMainWindow *parent)
    : QMainWindow(parent), logger_(this), xml_settings_(xml_settings)
{
	logger_.logSystemInfo();

	/* retrieve and set main window geometry */
	dim::ScreenSize screen;
	dim::MainDimension dim_inishell(screen);
	dim::setSize(this, dim_inishell.getDimension());
	dim::LogDimension dim_logger(screen);
	dim::setSize(&logger_, dim_logger.getDimension());

	/* create basic GUI items */
	createMenu();
	createToolbar();
	createStatusbar();
	preview_ = new PreviewWindow(this);
	/* create the dynamic GUI area */
	control_panel_ = new MainPanel;
	this->setCentralWidget(control_panel_);

//	const QString xml_file("doc/examples/meteoio_config.xml");
	const QString xml_file("min.xml");
	if (QFile::exists(xml_file)) {
		XMLReader xml;
		QString xml_error;
		const bool success = xml.read(xml_file, xml_error);
		if (!success)
			Error(tr("Errors occured when parsing the XML configuration file"),
			    tr("File: \"") + xml_file + "\"", xml_error);
		setStatus("Building GUI...", "black", true);
		buildGui(xml.getXml());
		setStatus("Ready.", "sysinfo", false);
		control_panel_->getWorkflowPanel()->buildWorkflowPanel(xml.getXml());
	} else {
		Error(tr("Last opened XML file not found."), tr("Input file \"%1\" does not exist").arg(xml_file));
	}

	ini_.setLogger(&logger_);
	ini_.setFile("io.ini");
//	ini_.setFile("tests/unit_test_inireader.ini");
	setGuiFromIni(ini_);
}

/**
 * @brief Build the dynamic GUI.
 * This function initiates the recursive GUI building with an XML document that was
 * parsed beforehand.
 * @param[in] xml XML to build the gui from.
 */
void MainWindow::buildGui(const QDomDocument &xml)
{
	QDomNode root = xml.firstChild();
	 //no parent group - tabs will be created for top level:
	recursiveBuild(root, nullptr, QString());
}

bool MainWindow::setGuiFromIni(const INIParser &ini)
{
	for (auto &sec : ini.getSections())
	{
		ScrollPanel *tab_scroll = getControlPanel()->getSectionScrollarea(sec.getName(), true);
		if (tab_scroll != nullptr) { //section exists on GUI
			for (auto &kv : sec.getKeyValueList() ) {
				QWidgetList widgets = findPanel(tab_scroll, sec, kv.second);
				if (!widgets.isEmpty()) {
					for (int ii = 0; ii < widgets.size(); ++ii) //multiple panels can share the same key
						widgets.at(ii)->setProperty("ini_value", kv.second.getValue());
				} else {
					logger_.log(tr("No GUI element found for INI key \"") + sec.getName() + Cst::sep + kv.second.getKey() + "\"",
					    colors::getQColor("iniwarning"));
				}
			} //endfor kv
		} else { //section does not exist
			//TODO: save to unused keys
			log(tr("No matching section in GUI for section from INI file: \"") + sec.getName() + "\"",
			    colors::getQColor("iniwarning"));
		} //endif section exists
	} //endfor it
	return true;
}

QWidgetList MainWindow::findPanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QWidgetList panels;
	panels = findSimplePanel(parent, section, keyval);
	int panel_count = parent->findChildren<Atomic *>().size();
	if (panels.isEmpty()) {
		panels = prepareSelector(parent, section, keyval); //TODO: logic cleanup
	}
	if (panels.isEmpty()) {
		panels = prepareReplicator(parent, section, keyval);
	}
	if (parent->findChildren<Atomic *>().size() != panel_count) //new elements were created
		return findPanel(parent, section, keyval);
	else
		return panels;
}

QWidgetList MainWindow::findSimplePanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	/* simple, not nested, keys */
	return parent->findChildren<QWidget *>(Atomic::getQtKey(id));
}

QWidgetList MainWindow::prepareSelector(QWidget *parent, const Section &section, const KeyValue &keyval)
{ //TODO: move to gui_elements.cc
	QString id(section.getName() + Cst::sep + keyval.getKey());
	const QString regex_selector("^" + section.getName() + Cst::sep + R"((\w+)(::)(\w+?)([0-9]*$))");
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.hasMatch()) { //it's from a selector with template children
		//try to find selector:
		static const size_t idx_parameter = 1;
		static const size_t idx_keyname = 3;
		static const size_t idx_optional_number = 4;
		const QString parameter(matches.captured(idx_parameter));
		const QString key_name(matches.captured(idx_keyname));
		const QString number(matches.captured(idx_optional_number));
		QString gui_id = section.getName() + Cst::sep + "%" + Cst::sep + key_name + (number.isEmpty()? "" : "#");
		QList<Selector *> selector_list = parent->findChildren<Selector *>(Atomic::getQtKey(gui_id));
		for (int ii = 0; ii < selector_list.count(); ++ii)
			selector_list.at(ii)->setProperty("ini_value", parameter);
		//now all the right child widgets should exist and we can try to find again:
		const QString key_id = section.getName() + Cst::sep + keyval.getKey(); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));

	}
	return QWidgetList();
}

QWidgetList MainWindow::prepareReplicator(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	const QString regex_selector("^" + section.getName() + Cst::sep + R"((\w+)(::)*(\w+)?([0-9]+)$)"); //TODO: allow :: only if not end of line
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.hasMatch()) { //it's from a replicator with template children
		//try to find replicator:
		static const size_t idx_name = 1;
		static const size_t idx_parameter = 3;
		static const size_t idx_number = 4;
		const QString name(matches.captured(idx_name));
		const QString parameter(matches.captured(idx_parameter));
		const QString number(matches.captured(idx_number));
		QString gui_id = section.getName() + Cst::sep + name;
		if (!parameter.isEmpty())
			gui_id += Cst::sep + parameter;
		gui_id += "#";

		QList<Replicator *> replicator_list = parent->findChildren<Replicator *>(Atomic::getQtKey(gui_id));
		//TODO/BUG: we can't initiate a .replicate event and set an INI value the same way, this is a bug.
		//Best be fixed when the replicator can handle the numbers better.
		for (int ii = 0; ii < replicator_list.count(); ++ii)
			replicator_list.at(ii)->setProperty("ini_value", number);
		//now all the right child widgets should exist and we can try to find again:
		const QString key_id = section.getName() + Cst::sep + keyval.getKey(); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));
	} //endif has match
	return QWidgetList();
}

/**
 * @brief Build the main window's menu items.
 */
void MainWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_quit = new QAction(tr("&Exit"));
	menu_file->addAction(file_quit);
	connect(file_quit, &QAction::triggered, this, &MainWindow::quitProgram);

	/* View menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	auto *view_preview = new QAction(tr("&Preview"));
	menu_view->addAction(view_preview);
	connect(view_preview, &QAction::triggered, this, &MainWindow::viewPreview);
	auto *view_log = new QAction(tr("&Log"));
	menu_view->addAction(view_log);
	connect(view_log, &QAction::triggered, this, &MainWindow::viewLogger);
	auto *view_settings = new QAction(tr("&Settings"));
	menu_view->addAction(view_settings);
	connect(view_settings, &QAction::triggered, this, &MainWindow::viewSettings);

	/* Help menu */
	QMenu *menu_help = this->menuBar()->addMenu(tr("&Help")); //TODO: move to the right
	auto *help_about = new QAction(tr("&About"));
	menu_help->addAction(help_about);
	connect(help_about, &QAction::triggered, this, &MainWindow::helpAbout);
}

/**
 * @brief Create the main window's toolbar.
 */
void MainWindow::createToolbar()
{
	//TODO: store position
	toolbar_ = this->addToolBar("main");
	QPixmap file_open(":/icons/file_open.png");
	toolbar_->addAction(QIcon(file_open), tr("Open File"));
}

void MainWindow::createStatusbar()
{
	auto *spacer_widget = new QWidget;
	spacer_widget->setFixedSize(5, 0);
	status_label_ = new QLabel;
	status_icon_ = new QLabel;
	statusBar()->addWidget(spacer_widget);
	statusBar()->addWidget(status_label_);
	statusBar()->addPermanentWidget(status_icon_);
}

/**
 * @brief Display a text in the main window's status bar.
 * @details Events such as hovering over a menu may clear the status. It is meant
 * for temporary messages.
 * @param[in] message The text to display.
 * @param[in] time Time to display the text for.
 */
void MainWindow::setStatus(const QString &message, const QString &color, const bool &status_light, const int &time)
{ //only temporary messages are possible via statusBar()->showMessage
	status_label_->setText(message);
	status_label_->setStyleSheet("QLabel {color: " + colors::getQColor(color).name() + "}");
	setStatusLight(status_light);
	if (time > 0) {
		status_timer_.stop();
		status_timer_.singleShot(time, this, &MainWindow::clearStatus);
	}
}

void MainWindow::setStatusLight(const bool &on)
{
	const QPixmap icon((on? "resources/icons/active.png" : "resources/icons/inactive.png"));
	status_icon_->setPixmap(icon.scaled(15, 15));
}

void MainWindow::clearStatus()
{
	status_label_->setText(QString());
}

/**
 * @brief Event handler for the "File::Quit" menu: quit the main program.
 */
void MainWindow::quitProgram()
{
	QApplication::quit();
}

void MainWindow::viewPreview()
{
	preview_->addIniTab();
	preview_->show();
	preview_->raise();
}

/**
 * @brief Event handler for the "View::Log" menu: open the Logger window.
 */
void MainWindow::viewLogger()
{
	logger_.show(); //the logger is always kept in scope
	logger_.raise(); //bring to front
}

/**
 * @brief Event handler for the "View::Settings" menu: open the settings dialog.
 */
void MainWindow::viewSettings()
{
	static auto *settings_dialog = new Settings(this); //allow only once, load when asked
	settings_dialog->show();
	settings_dialog->raise();
}

void MainWindow::helpAbout()
{
	static auto *about = new AboutWindow(this);
	about->show();
	about->raise();
}
