/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "MainPanel.h"
#include "src/main/constants.h"
#include "src/gui_elements/gui_elements.h"
#include "src/gui/WorkflowPanel.h"

#include <QGroupBox>
#include <QHBoxLayout>
#include <QListWidget>
#include <QSplitter>

#ifdef DEBUG
	#include <iostream>
#endif

/**
 * @class ScrollPanel
 * @brief The main scroll panel controlling all dynamically built panels.
 * @param[in] section The section this panel corresponds to. It could be used for styling but is
 * unused otherwise.
 * @param[in] parent The parent widget (the main tab bar).
 */
ScrollPanel::ScrollPanel(const QString &section, QWidget *parent)
    : Atomic(section, "_main_scroll_panel", parent)
{
	/* create a scroll area and put a Group in it */
	main_area_ = new QScrollArea;
	setPrimaryWidget(main_area_);
	main_area_->setWidgetResizable(true); //best to let the manager do its job - troubles ahead if we don't
	main_area_->setStyleSheet("QScrollArea {border: none}"); //we have one from the tabs already
	main_group_ = new Group(section, "_scrollpanel_group_");
	main_area_->setWidget(main_group_);

	/* main layout */
	auto *layout = new QVBoxLayout;
	layout->addWidget(main_area_);
	this->setLayout(layout);
}

/**
 * @brief Retrieve the main grouping element of a scroll panel (there is one per tab).
 * @return The main Group of this scroll panel holding all widgets.
 */
Group * ScrollPanel::getGroup() const
{
	return this->main_group_;
}

/**
 * @class MainPanel
 * @brief Constructor for the main panel, i. e. the tab widget with scroll area children.
 * @param[in] parent The parent widget (the main window).
 */
MainPanel::MainPanel(QWidget *parent) : QWidget(parent)
{
	/* widget(s) on the left side and main tab bar */
	workflow_panel_ = new WorkflowPanel;
	workflow_stack_ = new QStackedWidget;
	section_tab_ = new QTabWidget;
	workflow_stack_->addWidget(section_tab_);

	/* main layout */
	auto *main_layout = new QHBoxLayout;
	auto *splitter = new QSplitter;
	splitter->addWidget(workflow_panel_); //allow to resize with mouse
	splitter->addWidget(workflow_stack_);
	splitter->setStretchFactor(0, Cst::proportion_workflow_horizontal_percent);
	splitter->setStretchFactor(1, 100 - Cst::proportion_workflow_horizontal_percent);
	main_layout->addWidget(splitter);
	this->setLayout(main_layout);
}

/**
 * @brief Retrieve the ScrollPanel of a section.
 * @details Sections are grouped in tab bar elements. For this, they are put in that specific
 * tab panel's ScrollPanel which can be retrieved by this function (to build top level panels into).
 * If the section/tab does not exist yet it is created.
 * @param[in] section The INI section for which to get the corresponding ScrollPanel.
 * @return
 */
ScrollPanel * MainPanel::getSectionScrollarea(const QString &section, const bool&no_create)
{ //find or create section tab
	for (int ii = 0; ii < section_tab_->count(); ++ii) {
		if (section_tab_->tabBar()->tabText(ii).toLower() == section.toLower())
			return qobject_cast<ScrollPanel *>(section_tab_->widget(ii)); //cast to access members
	}
	if (!no_create) {
		auto *new_scroll = new ScrollPanel(section);
		section_tab_->addTab(new_scroll, section);
		return new_scroll;
	} else { //don't add scroll bars for keys found in ini files
		return nullptr;
	}
}
