/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Logger.h"
#include "src/main/colors.h"
#include "src/main/Error.h"

#include <QCoreApplication>
#include <QFileDialog>
#include <QGridLayout>
#include <QListWidgetItem>
#include <QPushButton>
#include <QTime>

/**
 * @class Logger
 * @brief Default constructor for the Logger.
 * @details This class is constructed as once and stays alive as a logging window.
 * @param[in] parent The parent window (should be main window).
 */
Logger::Logger(QWidget *parent) : QWidget(parent, Qt::Window) //dedicated window
{
	/* log list, close and save buttons */
	loglist_ = new QListWidget(this);
	auto *close_button = new QPushButton(tr("&Close"), this);
	connect(close_button, SIGNAL(clicked()), this, SLOT(closeLogger()));
	auto *save_button = new QPushButton(tr("&Save"), this);
	connect(save_button, SIGNAL(clicked()), this, SLOT(saveLog()));

	/* main layout */
	auto *log_layout = new QGridLayout(this);
	log_layout->addWidget(loglist_, 0, 0, 1, 2);
	log_layout->addWidget(close_button, 1, 0, Qt::AlignLeft);
	log_layout->addWidget(save_button, 1, 1, Qt::AlignRight);
	this->setLayout(log_layout);

	this->setWindowTitle(tr("Log Messages") + " ~ " + QCoreApplication::applicationName());
}

/**
 * @brief Add a text message to the Logger window.
 * @param[in] message The log message.
 * @param[in] color Color of the log message.
 */
void Logger::log(const QString &message, const QColor &color)
{
	QString timestamp( QTime::currentTime().toString("[hh:mm:ss]") );
	auto *msg = new QListWidgetItem(timestamp + " " + message, loglist_);
	msg->setForeground(color);
}

/**
 * @brief Log some system and Qt version info.
 */
void Logger::logSystemInfo()
{
	log(QCoreApplication::applicationName() + " " + QCoreApplication::applicationVersion() +
	    tr(" built with Qt ") + QT_VERSION_STR, colors::getQColor("sysinfo"));
	log(tr("Running on ") + QSysInfo::prettyProductName() + ", " + QSysInfo::kernelVersion() + ", "
	    + QSysInfo::buildAbi(), colors::getQColor("sysinfo"));
}

/**
 * @brief Event listener for the save button: save the log to a file.
 */
void Logger::saveLog()
{
	const QString filename = QFileDialog::getSaveFileName(this, tr("Save Log"), "./",
	    tr("Text Files") + " (*.log *.txt *.dat);;" + tr("All Files") + " (*)", nullptr,
	    QFileDialog::DontUseNativeDialog); //native is all but smooth on GNOME
	//QTBUG: https://bugreports.qt.io/browse/QTBUG-67866?focusedCommentId=408617&page=com.atlassian.jira.plugin.system.issuetabpanels:comment-tabpanel#comment-408617

	//TODO: HTML and text logs
}

/**
 * @brief Event listener for the close button: close the window.
 * @details The window is closed but stays in scope.
 */
void Logger::closeLogger()
{
	this->close();
}
