/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "XMLReader.h"
#include "src/main/colors.h"
#include "src/main/common.h"
#include "src/main/inishell.h"

#include <QCoreApplication> //for translations
#include <QtXmlPatterns/QXmlSchema>
#include <QtXmlPatterns/QXmlSchemaValidator>

#ifdef DEBUG
	#include <QDebug>
	#include <iostream>
#endif

QDomNode prependParent(QDomNode &child)
{
	QByteArray parent_array("<dummy_parent></dummy_parent>");
	QDomDocument parent_doc;
	parent_doc.setContent(parent_array, false);
	parent_doc.firstChild().appendChild(child.cloneNode(true));
	return parent_doc.firstChild();
}

XMLReader::XMLReader(const QString &filename, QString &xml_error)
{
	this->read(filename, xml_error);
}

void XMLReader::read(const QString &filename, QString &xml_error, const bool &no_references)
{
	QFile infile(filename);
	QFileInfo finfo(infile);
	master_xml_path_ = finfo.path();
	if (!infile.open(QIODevice::ReadOnly | QIODevice::Text)) {
		xml_error = QApplication::tr(R"(Could not open file "%1" for reading)").arg(filename) + "\n";
		return;
	}
	this->read(infile, xml_error, no_references);
	infile.close();
}

void XMLReader::read(QFile &file, QString &xml_error, const bool &is_settings_file)
{
	QString error_msg;
	int error_line, error_column;
	xml_error = QString();

	if (!xml_.setContent(&file, false, &error_msg, &error_line, &error_column)) {
		xml_error = QString(QCoreApplication::tr(
		    "%1 (line %2, column %3)")).arg(error_msg).arg(error_line).arg(error_column) + "\n";
	}

	if (is_settings_file) //e. g. static GUI settings file
		return;

	parseIncludes(xml_error);
	validateSchema(xml_error);
	parseReferences();
}

void XMLReader::parseReferences()
{ //TODO: done quick & dirty
	QDomNodeList par_groups = getXml().elementsByTagName("parametergroup");
	QDomNodeList to_substitute = getXml().elementsByTagName("reference");
	for (int ii = 0; ii < to_substitute.size(); ++ii) {
		const QString sub_name(to_substitute.at(ii).toElement().attribute("name"));
		for (int jj = 0; jj < par_groups.size(); ++jj) { //check nodes we can refer to
			if (par_groups.at(jj).toElement().attribute("name").toLower() == sub_name.toLower()) {
				for (QDomNode child = par_groups.at(jj).firstChild(); !child.isNull(); child = child.nextSibling())
					to_substitute.at(ii).parentNode().appendChild(child.cloneNode()); //TODO: shallow copy
			}
		}
	} //endfor to_substitute
}

void XMLReader::parseIncludes(QString &xml_error)
{
	while (true) {
		QDomNodeList includes = xml_.elementsByTagName("include");
		if (includes.isEmpty())
			break;
		for (int ii = 0; ii < includes.count(); ++ii) {
			const QString include_file_name(includes.at(ii).toElement().attribute("file"));
			QFile include_file(master_xml_path_ + "/" + include_file_name);
			if (include_file.open(QIODevice::ReadOnly)) {
				QDomDocument inc;
				QString error_msg;
				int error_line, error_column;
				if (!inc.setContent(include_file.readAll(), false, &error_msg, &error_line, &error_column)) {
					xml_error += QString(QCoreApplication::tr(
					    R"([Include file "%1"] %2 (line %3, column %4))")).arg(include_file_name).arg(
					    error_msg).arg(error_line).arg(error_column) + "\n";
				}
				QDomDocumentFragment new_fragment = inc.createDocumentFragment();
				for (QDomNode nd = inc.firstChild().firstChild(); !nd.isNull(); nd = nd.nextSibling())
					new_fragment.appendChild(nd.cloneNode(true));
				const QDomNode success = includes.at(ii).parentNode().replaceChild(new_fragment, includes.at(ii));
#ifdef DEBUG
				if (success.isNull()) {
					qDebug() << "Replacing a node failed for inclusion system in file " << include_file_name << endl;
					return; //endless loop otherwise
				}
#endif //def DEBUG
			} else {
				topLog(QString(QCoreApplication::tr("Unable to open XML include file \"%1\" for reading").arg(include_file_name)),
				    "xmlerror");
				return;
			}
		}
	} //end while
}

QDomDocument XMLReader::getXml() const
{
	return xml_;
}

void XMLReader::validateSchema(QString &xml_error)
{
	MessageHandler msgHandler;
	QXmlSchema schema;
	schema.setMessageHandler(&msgHandler);
	schema.load(QUrl("qrc:config_schema.xsd"));
	if (!schema.isValid()) {
		xml_error += QCoreApplication::tr("There is an error in the internal xsd file. Skipping schema validation.") + "\n";
	} else {
		QXmlSchemaValidator validator(schema);
		bool validation_success;
		validation_success = validator.validate(xml_.toString().toLocal8Bit());
		if (!validation_success) {
			QTextDocument error_msg;
			error_msg.setHtml(msgHandler.status()); //strip HTML
			xml_error += "[Schema validation failed] " + error_msg.toPlainText() +
			    QString(" (line %1, column %2)").arg(msgHandler.line()).arg(msgHandler.column()) + "\n";
		}
	}
}
