/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Error.h"
#include "colors.h"
#include "inishell.h"

#include <QCoreApplication>
#include <QMessageBox>
#include <QTextStream>

Error::Error(const QString &message)
{
	messageBox(message);
}

Error::Error(const QString &message, const QString &infotext)
{
	messageBox(message, infotext);
}

Error::Error(const QString &message, const QString &infotext, const QString &details)
{
	messageBox(message, infotext, details);
}

Error::Error(const QString &message, const QString &infotext, const QString &details,
    const error::urgency &level, const bool &no_log)
{
	messageBox(message, infotext, details, level, no_log);
	if (level == error::fatal) {
		QString msg;
		QTextStream ss(&msg);
		ss << QMessageBox::tr("Aborted after fatal error:") << endl;
		ss << message << endl;
		ss << infotext << endl << details;
		throw std::runtime_error(msg.toStdString());
		//TODO: autosave log?
	}
}

Info::Info(const QString &message)
{
	messageBox(message, QString(), QString(), error::info);
}

int messageBox(const QString &message, const QString &infotext, const QString &details,
    const error::urgency &level, const bool &no_log)
{
	QMessageBox msgBox;
	msgBox.setText("<b>" + message + "</b>");
	msgBox.setInformativeText(infotext);
	if (!details.isEmpty()) {
		msgBox.setDetailedText(details);
		msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Close);
		msgBox.setDefaultButton(QMessageBox::Close);
	}
	msgBox.setDefaultButton(QMessageBox::Ok);
	QString color("msg");
	QString title;
	switch (level) {
	case error::info:
		title = QMessageBox::tr("Info");
		msgBox.setIcon(QMessageBox::Information);
		color = "info";
		break;
	case error::warning:
		title = QMessageBox::tr("Warning");
		msgBox.setIcon(QMessageBox::Warning);
		color = "warning";
		break;
	case error::error:
		title = QMessageBox::tr("Error");
		msgBox.setIcon(QMessageBox::Critical);
		color = "error";
		break;
	case error::critical:
		title = QMessageBox::tr("Critical Error");
		msgBox.setIcon(QMessageBox::Critical);
		color = "criticalError";
		break;
	case error::fatal:
		title = QMessageBox::tr("Fatal Error");
		msgBox.setIcon(QMessageBox::Critical);
		color = "fatalError";
	}
	if (!no_log) //for if we want to log something different than is displayed in the error dialog
		topLog(title + ": " + message + (infotext.isEmpty()? "" : " ~ " + infotext) +
		    (details.isEmpty()? "" : " ~ " + details), color);
	msgBox.setWindowTitle(title + " ~ " + QCoreApplication::applicationName());
	return msgBox.exec();
}
