﻿/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "WorkflowPanel.h"
#include "src/gui_elements/Atomic.h" //for getQtKey()
#include "src/main/colors.h"
#include "src/main/constants.h"
#include "src/main/inishell.h"

#include <QDateTimeEdit>
#include <QDesktopServices>
#include <QGroupBox>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QRegularExpression>
#include <QString>
#include <QStringList>
#include <QVBoxLayout>

#ifdef DEBUG
	#include <iostream>
#endif

namespace html {

QString bold(const QString &text)
{
	return "<b>" + text + "</b>";
}

QString color(const QString &text, const QString &color)
{
	QString strRet( "<font color=\"" + colors::getQColor(color).name() + "\">" + text + "</font>");
	return strRet;
}

} //namespace html

WorkflowPanel::WorkflowPanel(QWidget *parent) : QWidget(parent)
{
	this->setMaximumWidth(Cst::width_workflow_max); //to stop the splitter from making it huge
	workflow_container_ = new QToolBox;
	connect(workflow_container_, &QToolBox::currentChanged, this, &WorkflowPanel::toolboxClicked);
	applications_ = new ApplicationsView;
	filesystem_ = new FolderView;
	workflow_container_->addItem(applications_, tr("Applications"));
	//: Translation hint: This is for the workflow panel on the left side.
	workflow_container_->addItem(filesystem_, tr("INI files"));

	auto *layout = new QVBoxLayout;
	layout->addWidget(workflow_container_);
	this->setLayout(layout);
}

void WorkflowPanel::buildWorkflowPanel(const QDomDocument &xml)
{
	QDomNode workroot = xml.firstChildElement().firstChildElement("workflow");
	for (QDomElement work = workroot.toElement().firstChildElement("section"); !work.isNull();
	    work = work.nextSiblingElement("section")) {
		buildWorkflowSection(work);
	}
}

void WorkflowPanel::clearXmlPanels()
{
	for (int ii = 0; ii < workflow_container_->count(); ++ii) {
		if (workflow_container_->widget(ii)->property("from_xml").toBool() == true) {
			workflow_container_->widget(ii)->deleteLater();
		}
	}
}

QString WorkflowPanel::getCurrentApplication() const
{
	return applications_->getCurrentApplication();
}

//QSize WorkflowPanel::sizeHint() const
//{
//	return {300, 1};
//}

void WorkflowPanel::buildWorkflowSection(QDomElement &section)
{
	auto *workflow_frame = new QFrame;
	auto *workflow_layout = new QVBoxLayout;
	const QString caption(section.attribute("caption"));
	for (QDomElement el = section.firstChildElement("element"); !el.isNull(); el = el.nextSiblingElement("element")) {
		QWidget * section_item = workflowElementFactory(el);
		workflow_frame->setProperty("from_xml", true); //to be able to delete all dynamic ones
		if (section_item != nullptr) {
			section_item->setParent(workflow_frame); //for once this is important (to check which terminal view is associated with a button)
			workflow_layout->addWidget(section_item, 0, Qt::AlignTop);
			if (section_item->property("action").toString() == "terminal")
				workflow_frame->setProperty("action", "terminal");
		} else {
			topLog(tr(R"(Workflow element "%1" unknown)").arg(el.attribute("type")), "xmlerror");
		}
	}

	if (workflow_frame->property("action").toString() == "terminal") {
		TerminalView *new_terminal = new TerminalView;
		workflow_frame->setProperty("stack_index", getMainWindow()->getControlPanel()->getWorkflowStack()->count());
		getMainWindow()->getControlPanel()->getWorkflowStack()->addWidget(new_terminal);
	}
	auto *spacer = new QSpacerItem(-1, -1, QSizePolicy::Expanding, QSizePolicy::Expanding);
	workflow_layout->addSpacerItem(spacer);
	workflow_frame->setLayout(workflow_layout);
	workflow_container_->addItem(workflow_frame, caption);
}

QWidget * WorkflowPanel::workflowElementFactory(QDomElement &item)
{
	QWidget *element = nullptr;
	const QString identifier(item.attribute("type"));
	const QString id(item.attribute("id"));
	const QString caption(item.attribute("caption"));
	if (identifier == "datetime") {
		QString tmp = item.attribute("default");
		QDateTime default_date(QDateTime::fromString(item.attribute("default"), Qt::ISODate));
		if (!default_date.isValid())
			default_date = QDateTime::currentDateTime();
		element = new QDateTimeEdit(default_date);
	} else if (identifier == "button") {
		element = new QPushButton(caption);

		static const QString regex_openurl(R"(openurl\((.*)\))");
		static const QRegularExpression rex(regex_openurl);
		QStringList action_list;
		for (QDomElement cmd = item.firstChildElement("command"); !cmd.isNull(); cmd = cmd.nextSiblingElement("command")) {
			action_list.push_back(cmd.text());
			QRegularExpressionMatch url_match = rex.match(cmd.text());
			if (!url_match.hasMatch())
				element->setProperty("action", "terminal");
		}
		connect(static_cast<QPushButton *>(element), &QPushButton::clicked, this,
		    [=] { buttonClicked(static_cast<QPushButton *>(element), action_list); });

	} else if (identifier == "label") {
		element = new QLabel(caption);
		element->setStyleSheet("QLabel {color: " + colors::getQColor("key").name() + "}");
		static_cast<QLabel *>(element)->setWordWrap(true);
	} else if (identifier == "text") {
		element = new QLineEdit;
		static_cast<QLineEdit *>(element)->setText(item.attribute("default"));
	}
	if (element) {
		element->setObjectName("_workflow_" + Atomic::getQtKey(id));
	}
	return element;
}

void WorkflowPanel::buttonClicked(QPushButton *button, const QStringList &action_list)
{
	for (int ii = 0; ii < action_list.size(); ++ii) {
		QString command(parseCommand(action_list[ii]));

		static const QString regex_openurl(R"(openurl\((.*)\))");
		static const QRegularExpression rex(regex_openurl);
		QRegularExpressionMatch url_match = rex.match(action_list[ii]);
		if (url_match.hasMatch()) {
			QDesktopServices::openUrl(QUrl(url_match.captured(1)));
			continue;
		}

		const int idx = button->parent()->property("stack_index").toInt();
		TerminalView *terminal = static_cast<TerminalView *>(getMainWindow()->getControlPanel()->getWorkflowStack()->widget(idx));

		QProcess *process = new QProcess(this);

		connect(process, QOverload<int, QProcess::ExitStatus>::of(&QProcess::finished), this,
			[=](int exit_code, QProcess::ExitStatus exit_status) { processFinished(exit_code, exit_status, terminal); });
		connect(process, &QProcess::readyReadStandardOutput, this, [=]() { processStandardOutput(terminal); });
		connect(process, &QProcess::readyReadStandardError, this, [=]() { processStandardError(terminal); });

		process->start(command);
		terminal->log(html::bold("$ " + command));
		if (ii != action_list.size() - 1) {
			topStatus(tr("Waiting for process to finish..."), "black", true);
			getMainWindow()->refreshStatus();
			getMainWindow()->repaint();
			process->waitForFinished(10*60*1000);
			topStatus(QString(), QString(), false);
		} else {
			topStatus("A process is running...", "black", true);
		}
	}

	//"<A HREF=\"full.html?file=%s/%s.smet\" target=\"_blank\">%s</A>"
}

void WorkflowPanel::toolboxClicked(int index)
{
	if (getMainWindow()->getControlPanel() == nullptr)
		return; //stacked widget not built yet
	QStackedWidget *stack = getMainWindow()->getControlPanel()->getWorkflowStack();
	const QString action(workflow_container_->widget(index)->property("action").toString());
	if (action == "terminal") {
		int idx = workflow_container_->widget(index)->property("stack_index").toInt();
		stack->setCurrentIndex(idx);
	} else {
		getMainWindow()->getControlPanel()->getWorkflowStack()->setCurrentIndex(0);
	}
}

QString WorkflowPanel::parseCommand(const QString &action)
{
	QString command(action);
	static const QString regex_substitution(R"((?<=^|\s)%\w+(?=\s|$))");
	static const QRegularExpression rex(regex_substitution);

	QRegularExpressionMatchIterator rit = rex.globalMatch(action);
	while (rit.hasNext()) {
		QRegularExpressionMatch match = rit.next();
		const QString id(match.captured(0).mid(1));
		QWidget *input_widget = this->findChild<QWidget *>("_workflow_" + Atomic::getQtKey(id));
		if (input_widget) {
			const QString substitution = getWidgetValue(input_widget);
			command.replace(match.captured(0), substitution);
		}
	} //end while rit.hasNext()
	return command;
}

QString WorkflowPanel::getWidgetValue(QWidget *widget) const
{
	if (auto *wid = qobject_cast<QDateTimeEdit *>(widget))
		return wid->dateTime().toString(Qt::DateFormat::ISODate);
	else if (auto *wid = qobject_cast<QLineEdit *>(widget))
		return wid->text().replace("%lastini", getMainWindow()->getIni()->getFilename());
	return QString();
}

void WorkflowPanel::processFinished(int exit_code, QProcess::ExitStatus exit_status, TerminalView *terminal)
{
	if (exit_status != QProcess::NormalExit) {
		const QString msg(QString(tr(
		    "The process was terminated unexpectedly (exit code: %1, exit status: %2).")).arg(exit_code).arg(exit_status));
		terminal->log(html::color(html::bold(msg), "fatalerror"));
	}
	terminal->log(html::color(html::bold("$ " + QDir::currentPath()), "sysinfo"));
	topStatus(tr("Process has finished"), "black", false, Cst::msg_length);
}

void WorkflowPanel::processStandardOutput(TerminalView *terminal)
{
	const QString str_std(static_cast<QProcess *>(QObject::sender())->readAllStandardOutput());
	if (!str_std.isEmpty())
		terminal->log(str_std);
}

void WorkflowPanel::processStandardError(TerminalView *terminal)
{
	const QString str_err(static_cast<QProcess *>(QObject::sender())->readAllStandardError());
	if (!str_err.isEmpty())
		terminal->log(html::color(html::bold(str_err), "error"));
	topStatus(tr("Process was terminated"), "error", false, Cst::msg_length);
}
