/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Settings.h"
#include "src/main/constants.h"
#include "src/main/Error.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QApplication>
#include <QCheckBox>
#include <QCoreApplication>
#include <QGridLayout>
#include <QGroupBox>
#include <QLabel>
#include <QPushButton>
#include <QRadioButton>

panelFiles::panelFiles(QWidget *parent) : QWidget(parent)
{
	auto *layout = new QGridLayout(this);
	auto *file_list = new QListWidget(this);
	new QListWidgetItem("meteoio.xml", file_list);
	layout->addWidget(file_list, 0, 0, Qt::AlignTop);
	this->setLayout(layout);
}

panelInireader::panelInireader(QWidget *parent) : QWidget(parent)
{
	auto *layout = new QVBoxLayout;

	auto *whitespace_box = new QGroupBox("Whitespace treatment");
	auto *whitespace_layout = new QVBoxLayout;
	QButtonGroup whitespace_buttongroup;
	auto *whitespace_button_leave = new QRadioButton("Leave whitespaces in place");
	whitespace_layout->addWidget(whitespace_button_leave);
	auto *whitespace_button_off = new QRadioButton("Remove duplicate whitespaces");
	whitespace_layout->addWidget(whitespace_button_off);
	auto *whitespace_button_tabs = new QRadioButton("Fill with tabs");
	groupButtons(whitespace_buttongroup, whitespace_box);
	whitespace_layout->addWidget(whitespace_button_tabs, 0, Qt::AlignTop);
	whitespace_box->setLayout(whitespace_layout);

	layout->addWidget(whitespace_box);
	this->setLayout(layout);
}

Settings::Settings(QWidget *parent) : QWidget(parent, Qt::Window)
{
	auto *layout = new QGridLayout;
	buildNavigationList(); //fill list with "buttons" for navigation

	pages_ = new QStackedWidget(this);
	pages_->addWidget(new panelFiles);
	pages_->addWidget(new panelInireader);

	layout->addWidget(navigation_list_, 0, 0, 1, 3);
	layout->addWidget(pages_, 1, 0, 1, 3);

	/* close & save buttons */
	auto *close_button = new QPushButton(tr("&Close"), this);
	connect(close_button, &QPushButton::clicked, this, &Settings::closeSettings);
	auto *apply_button = new QPushButton(tr("&Apply"), this);
	auto *save_button = new QPushButton(tr("&Save"), this);
	layout->addWidget(close_button, 2, 0, Qt::AlignLeft);
	layout->addWidget(apply_button, 2, 2, Qt::AlignLeft);
	layout->addWidget(save_button, 2, 2, Qt::AlignRight);

	this->setLayout(layout);
	this->setWindowTitle(tr("Settings") + " ~ " + QCoreApplication::applicationName());
	this->setFixedSize(Cst::width_settings, Cst::height_settings);
}

void Settings::buildNavigationList()
{
	navigation_list_ = new QListWidget(this);
	navigation_list_->setViewMode(QListView::IconMode);
	navigation_list_->setMovement(QListView::Static); //can not drag items
	navigation_list_->setSpacing(Cst::settings_navigation_spacing); //a little room around the items
	navigation_list_->setMaximumHeight(Cst::settings_navigation_maxheight);

	auto *item_files = new QListWidgetItem(navigation_list_);
	item_files->setIcon(QIcon(":/icons/xml_file.png"));
	item_files->setText(tr("Files"));
	auto *item_inireader = new QListWidgetItem(navigation_list_);
	item_inireader->setIcon(QIcon(":/icons/ini_file.png"));
	item_inireader->setText(tr("INI Reader"));

	navigation_list_->setCurrentRow(0); //highlight 1st
	connect(navigation_list_, &QListWidget::currentItemChanged, this, &Settings::switchPanel);
}

void Settings::switchPanel(QListWidgetItem *current, QListWidgetItem * /*previous*/)
{
	pages_->setCurrentIndex(navigation_list_->row(current));
}

void Settings::closeSettings()
{
	this->close();
}

void groupButtons(QButtonGroup &group, QWidget *parent)
{ //add all radio buttons of a widget to a button group --> only one can be selected
	QList<QRadioButton *> buttons_list = parent->findChildren<QRadioButton *>();
	for (int bb = 0; bb < buttons_list.size(); ++bb)
		group.addButton(buttons_list[bb], bb);
}

void checkSettings(QDomDocument &xml_settings)
{
	if (xml_settings.isNull()) {
		std::cout << "no settings" << std::endl;
	}

	QDomNode check;
	QDomNode auto_root = xml_settings.firstChild().firstChildElement("auto");
	if (auto_root.isNull())
		auto_root = xml_settings.firstChild().appendChild(xml_settings.createElement("auto"));
	check = auto_root.firstChildElement("autoload");
	if (check.isNull())
		auto_root.appendChild(xml_settings.createElement("autoload"));
	QDomNode history_root = auto_root.firstChildElement("history");
	if (history_root.isNull())
		history_root = auto_root.appendChild(xml_settings.createElement("history"));
	check = history_root.firstChildElement("ini_folder");
	if (check.isNull())
		history_root.appendChild(xml_settings.createElement("ini_folder"));

}

void saveSettings(const QDomDocument &xml_settings)
{
	const QString settings_file("inishell_settings.xml");
	QFile outfile(settings_file);
	if(!outfile.open( QIODevice::WriteOnly | QIODevice::Text)) {
		Error(QApplication::tr("Could not open settings file for writing"), QString(),
		    settings_file + ":\n" + outfile.errorString());
		return;
	}

	QTextStream out_ss(&outfile);
	out_ss << xml_settings.toString();
	outfile.close();
}
